/**
 * Types
 */
/// <reference types="webxr" />
import { WonderlandEngine } from './engine.js';
import { Emitter } from './utils/event.js';
import { Material } from './resources/material-manager.js';
import { ComponentProperty, PropertyRecord } from './property.js';
import { ImageLike, NumberArray, TypedArray, TypedArrayCtor } from './types.js';
import { Resource, ResourceType, SceneResource } from './resources/resource.js';
import { Prefab } from './prefab.js';
import { Scene } from './scene.js';
import { NativeComponents } from './component.js';
/**
 * Component constructor type.
 *
 * For more information, please have a look at the {@link Component} class.
 */
export type ComponentConstructor<T extends Component = Component> = {
    new (scene: Prefab, manager: number, id: number): T;
} & {
    _isBaseComponent: boolean;
    _propertyOrder: string[];
    TypeName: string;
    Properties: Record<string, ComponentProperty>;
    InheritProperties?: boolean;
    onRegister?: (engine: WonderlandEngine) => void;
    UpdateAfter?: ((engine: WonderlandEngine) => ComponentConstructor[]) | ComponentConstructor[];
    UpdateBefore?: ((engine: WonderlandEngine) => ComponentConstructor[]) | ComponentConstructor[];
};
/**
 * Component prototype interface.
 *
 * User component's should have the same structure.
 */
export interface ComponentProto {
    /**
     * Triggered after the component instantiation.
     * For more information, please have a look at {@link Component.init}.
     */
    init?: () => void;
    /**
     * Triggered after the component is activated for the first time.
     * For more information, please have a look at {@link Component.start}.
     */
    start?: () => void;
    /**
     * Triggered once per frame.
     * For more information, please have a look at {@link Component.update}.
     *
     * @param dt Delta time, time since last update.
     */
    update?: (dt: number) => void;
    /**
     * Triggered when the component goes from deactivated to activated.
     * For more information, please have a look at {@link Component.onActivate}.
     */
    onActivate?: () => void;
    /**
     * Triggered when the component goes from activated to deactivated.
     * For more information, please have a look at {@link Component.onDeactivate}.
     */
    onDeactivate?: () => void;
    /**
     * Triggered when the component is removed from its object.
     * For more information, please have a look at {@link Component.onDestroy}.
     *
     * @since 0.9.0
     */
    onDestroy?: () => void;
}
/**
 * Callback triggered on collision event.
 *
 * @param type Type of the event.
 * @param other Other component that was (un)collided with
 */
export type CollisionCallback = (type: CollisionEventType, other: PhysXComponent) => void;
/**
 * Wonderland Engine API
 * @namespace WL
 */
/**
 * Default set of logging tags used by the API.
 */
export declare enum LogTag {
    /** Initialization, component registration, etc... */
    Engine = 0,
    /** Scene loading */
    Scene = 1,
    /** Component init, update, etc... */
    Component = 2
}
/**
 * Collider type enum for {@link CollisionComponent}.
 */
export declare enum Collider {
    /**
     * **Sphere Collider**:
     *
     * Simplest and most performant collision shape. If this type is set on a
     * {@link CollisionComponent}, only the first component of
     * {@link CollisionComponent#extents} will be used to determine the radius.
     */
    Sphere = 0,
    /**
     * **Axis Aligned Bounding Box Collider**:
     *
     * Box that is always aligned to XYZ axis. It cannot be rotated but is more
     * efficient than {@link Collider.Box}.
     */
    AxisAlignedBox = 1,
    /**
     * **Aligned Bounding Box Collider**:
     *
     * Box that matches the object's rotation and translation correctly. This
     * is the least efficient collider and should only be chosen over
     * {@link Collider.Sphere} and {@link Collider.AxisAlignedBox} if really
     * necessary.
     */
    Box = 2
}
/**
 * Alignment type enum for {@link TextComponent}.
 */
export declare enum Alignment {
    /** Text start is at object origin */
    Left = 0,
    /** Text center is at object origin */
    Center = 1,
    /** Text end is at object origin */
    Right = 2
}
/**
 * Vertical alignment type enum for {@link TextComponent}.
 */
export declare enum VerticalAlignment {
    /** Text line is at object origin */
    Line = 0,
    /** Text middle is at object origin */
    Middle = 1,
    /** Text top is at object origin */
    Top = 2,
    /** Text bottom is at object origin */
    Bottom = 3
}
/**
 * Justification type enum for {@link TextComponent}.
 *
 * @deprecated Please use {@link VerticalAlignment} instead.
 */
export declare const Justification: typeof VerticalAlignment;
/**
 * Effect type enum for {@link TextComponent}.
 */
export declare enum TextEffect {
    /** Text is rendered normally */
    None = 0,
    /** Text is rendered with an outline */
    Outline = 1,
    /** Text is rendered with a drop shadow */
    Shadow = 2
}
/**
 * Wrap mode enum for {@link TextComponent}.
 *
 * @since 1.2.1
 */
export declare enum TextWrapMode {
    /** Text doesn't wrap automatically, only with explicit newline */
    None = 0,
    /** Text wraps at word boundaries */
    Soft = 1,
    /** Text wraps anywhere */
    Hard = 2,
    /** Text is cut off */
    Clip = 3
}
/**
 * Input type enum for {@link InputComponent}.
 */
export declare enum InputType {
    /** Head input */
    Head = 0,
    /** Left eye input */
    EyeLeft = 1,
    /** Right eye input */
    EyeRight = 2,
    /** Left controller input */
    ControllerLeft = 3,
    /** Right controller input */
    ControllerRight = 4,
    /** Left ray input */
    RayLeft = 5,
    /** Right ray input */
    RayRight = 6
}
/**
 * Projection type enum for {@link ViewComponent}.
 */
export declare enum ProjectionType {
    /** Perspective projection */
    Perspective = 0,
    /** Orthographic projection */
    Orthographic = 1
}
/**
 * Light type enum for {@link LightComponent}.
 */
export declare enum LightType {
    /** Point light */
    Point = 0,
    /** Spot light */
    Spot = 1,
    /** Sun light / Directional light */
    Sun = 2
}
/**
 * Animation state of {@link AnimationComponent}.
 */
export declare enum AnimationState {
    /** Animation is currently playing */
    Playing = 0,
    /** Animation is paused and will continue at current playback
     * time on {@link AnimationComponent#play} */
    Paused = 1,
    /** Animation is stopped */
    Stopped = 2
}
/**
 * Root motion mode of {@link AnimationComponent}.
 */
export declare enum RootMotionMode {
    /** Do nothing */
    None = 0,
    /** Move and rotate root with the delta of its motion */
    ApplyToOwner = 1,
    /** Store the motion to be retrieved by a JS script */
    Script = 2
}
/**
 * Rigid body force mode for {@link PhysXComponent#addForce} and {@link PhysXComponent#addTorque}.
 *
 * [PhysX API Reference](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxapi/files/structPxForceMode.html)
 */
export declare enum ForceMode {
    /** Apply as force */
    Force = 0,
    /** Apply as impulse */
    Impulse = 1,
    /** Apply as velocity change, mass dependent */
    VelocityChange = 2,
    /** Apply as mass dependent force */
    Acceleration = 3
}
/**
 * Collision callback event type.
 */
export declare enum CollisionEventType {
    /** Touch/contact detected, collision */
    Touch = 0,
    /** Touch/contact lost, uncollide */
    TouchLost = 1,
    /** Touch/contact with trigger detected */
    TriggerTouch = 2,
    /** Touch/contact with trigger lost */
    TriggerTouchLost = 3
}
/**
 * Rigid body {@link PhysXComponent#shape}.
 *
 * [PhysX SDK Guide](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/Geometry.html#geometry-types).
 */
export declare enum Shape {
    /** No shape. */
    None = 0,
    /** Sphere shape. */
    Sphere = 1,
    /** Capsule shape. */
    Capsule = 2,
    /** Box shape. */
    Box = 3,
    /** Plane shape. */
    Plane = 4,
    /** Convex mesh shape. */
    ConvexMesh = 5,
    /** Triangle mesh shape. */
    TriangleMesh = 6
}
/**
 * Mesh attribute enum.
 * @since 0.9.0
 */
export declare enum MeshAttribute {
    /** Position attribute, 3 floats */
    Position = 0,
    /** Tangent attribute, 4 floats */
    Tangent = 1,
    /** Normal attribute, 3 floats */
    Normal = 2,
    /** Texture coordinate attribute, 2 floats */
    TextureCoordinate = 3,
    /** Color attribute, 4 floats, RGBA, range `0` to `1` */
    Color = 4,
    /** Joint id attribute, 8 unsigned ints */
    JointId = 5,
    /** Joint weights attribute, 8 floats */
    JointWeight = 6,
    /** Secondary texture coordinate attribute, 2 floats */
    SecondaryTextureCoordinate = 7
}
/**
 * Graphics API enum.
 * @since 1.5.0
 */
export declare enum GraphicsApi {
    /** WebGL 2.0 */
    WebGL2 = 0,
    /** WebGPU */
    WebGPU = 1
}
/**
 * Provides access to a component instance of a specified component type.
 *
 * @example
 *
 * This is how you extend this class to create your own custom
 * component:
 *
 * ```js
 * import { Component, Type } from '@wonderlandengine/api';
 *
 * export class MyComponent extends Component {
 *     static TypeName = 'my-component';
 *     static Properties = {
 *         myBoolean: { type: Type.Boolean, default: false },
 *     };
 *     start() {}
 *     onActivate() {}
 *     onDeactivate() {}
 *     update(dt) {}
 * }
 * ```
 *
 * In a component, the scene can be accessed using `this.scene`:
 *
 * ```js
 * import { Component, Type } from '@wonderlandengine/api';
 *
 * export class MyComponent extends Component {
 *     static TypeName = 'my-component';
 *     start() {
 *         const obj = this.scene.addObject();
 *     }
 * }
 * ```
 */
export declare class Component {
    /**
     * Pack scene index and component id.
     *
     * @param scene Scene index.
     * @param id Component id.
     * @returns The packed id.
     *
     * @hidden
     */
    static _pack(scene: number, id: number): number;
    /**
     * `true` for every class inheriting from this class.
     *
     * @note This is a workaround for `instanceof` to prevent issues
     * that could arise when an application ends up using multiple API versions.
     *
     * @hidden
     */
    static readonly _isBaseComponent = true;
    /**
     * Fixed order of attributes in the `Properties` array.
     *
     * @note This is used for parameter deserialization and is filled during
     * component registration.
     *
     * @hidden
     */
    static _propertyOrder: string[];
    /**
     * Unique identifier for this component class.
     *
     * This is used to register, add, and retrieve components of a given type.
     */
    static TypeName: string;
    /**
     * Properties of this component class.
     *
     * Properties are public attributes that can be configured via the
     * Wonderland Editor.
     *
     * Example:
     *
     * ```js
     * import { Component, Type } from '@wonderlandengine/api';
     * class MyComponent extends Component {
     *     static TypeName = 'my-component';
     *     static Properties = {
     *         myBoolean: { type: Type.Boolean, default: false },
     *         myFloat: { type: Type.Float, default: false },
     *         myTexture: { type: Type.Texture, default: null },
     *     };
     * }
     * ```
     *
     * Properties are automatically added to each component instance, and are
     * accessible like any JS attribute:
     *
     * ```js
     * // Creates a new component and set each properties value:
     * const myComponent = object.addComponent(MyComponent, {
     *     myBoolean: true,
     *     myFloat: 42.0,
     *     myTexture: null
     * });
     *
     * // You can also override the properties on the instance:
     * myComponent.myBoolean = false;
     * myComponent.myFloat = -42.0;
     * ```
     *
     * #### References
     *
     * Reference types (i.e., mesh, object, etc...) can also be listed as **required**:
     *
     * ```js
     * import {Component, Property} from '@wonderlandengine/api';
     *
     * class MyComponent extends Component {
     *     static Properties = {
     *         myObject: Property.object({required: true}),
     *         myAnimation: Property.animation({required: true}),
     *         myTexture: Property.texture({required: true}),
     *         myMesh: Property.mesh({required: true}),
     *     }
     * }
     * ```
     *
     * Please note that references are validated **once** before the call to {@link Component.start} only,
     * via the {@link Component.validateProperties} method.
     */
    static Properties: Record<string, ComponentProperty>;
    /**
     * When set to `true`, the child class inherits from the parent
     * properties, as shown in the following example:
     *
     * ```js
     * import {Component, Property} from '@wonderlandengine/api';
     *
     * class Parent extends Component {
     *     static TypeName = 'parent';
     *     static Properties = {parentName: Property.string('parent')}
     * }
     *
     * class Child extends Parent {
     *     static TypeName = 'child';
     *     static Properties = {name: Property.string('child')}
     *     static InheritProperties = true;
     *
     *     start() {
     *         // Works because `InheritProperties` is `true`.
     *         console.log(`${this.name} inherits from ${this.parentName}`);
     *     }
     * }
     * ```
     *
     * @note Properties defined in descendant classes will override properties
     * with the same name defined in ancestor classes.
     *
     * Defaults to `true`.
     */
    static InheritProperties?: boolean;
    /**
     * Called when this component class is registered.
     *
     * @example
     *
     * This callback can be used to register dependencies of a component,
     * e.g., component classes that need to be registered in order to add
     * them at runtime with {@link Object3D.addComponent}, independent of whether
     * they are used in the editor.
     *
     * ```js
     * class Spawner extends Component {
     *     static TypeName = 'spawner';
     *
     *     static onRegister(engine) {
     *         engine.registerComponent(SpawnedComponent);
     *     }
     *
     *     // You can now use addComponent with SpawnedComponent
     * }
     * ```
     *
     * @example
     *
     * This callback can be used to register different implementations of a
     * component depending on client features or API versions.
     *
     * ```js
     * // Properties need to be the same for all implementations!
     * const SharedProperties = {};
     *
     * class Anchor extends Component {
     *     static TypeName = 'spawner';
     *     static Properties = SharedProperties;
     *
     *     static onRegister(engine) {
     *         if(navigator.xr === undefined) {
     *             /* WebXR unsupported, keep this dummy component *\/
     *             return;
     *         }
     *         /* WebXR supported! Override already registered dummy implementation
     *          * with one depending on hit-test API support *\/
     *         engine.registerComponent(window.HitTestSource === undefined ?
     *             AnchorWithoutHitTest : AnchorWithHitTest);
     *     }
     *
     *     // This one implements no functions
     * }
     * ```
     */
    static onRegister?: (engine: WonderlandEngine) => void;
    /**
     * Component of classes that should be executed **before** this type.
     *
     * ## Examples
     *
     * ```js
     * class First extends Component {
     *     update() {
     *         console.log('Hello');
     *     }
     * }
     *
     * class Second extends Component {
     *     static UpdateAfter = [First];
     *     update() {
     *         console.log('World!');
     *     }
     * }
     * ```
     *
     * Will print:
     * - 'Hello'
     * - 'World!'
     */
    static UpdateAfter?: ((engine: WonderlandEngine) => ComponentConstructor[]) | ComponentConstructor[];
    /**
     * Component of classes that should be executed **after** this type.
     *
     * ## Examples
     *
     * ```js
     * class Second extends Component {
     *     update() {
     *         console.log('World!');
     *     }
     * }
     *
     * class First extends Component {
     *     static UpdateBefore = [Second];
     *     update() {
     *         console.log('Hello');
     *     }
     * }
     * ```
     *
     * Will print:
     * - 'Hello'
     * - 'World!'
     */
    static UpdateBefore?: ((engine: WonderlandEngine) => ComponentConstructor[]) | ComponentConstructor[];
    /**
     * Allows to inherit properties directly inside the editor.
     *
     * @note Do not use directly, prefer using {@link inheritProperties}.
     *
     * @hidden
     */
    static _inheritProperties(): void;
    /**
     * Triggered when the component is initialized by the runtime. This method
     * will only be triggered **once** after instantiation.
     *
     * @note During the initialization phase, `this.scene` will not match
     * `engine.scene`, since `engine.scene` references the **active** scene:
     *
     * ```js
     * import {Component} from '@wonderlandengine/api';
     *
     * class MyComponent extends Component{
     *     init() {
     *         const activeScene = this.engine.scene;
     *         console.log(this.scene === activeScene); // Prints `false`
     *     }
     *     start() {
     *         const activeScene = this.engine.scene;
     *         console.log(this.scene === activeScene); // Prints `true`
     *     }
     * }
     * ```
     */
    init?(): void;
    /**
     * Triggered when the component is started by the runtime, or activated.
     *
     * You can use that to re-initialize the state of the component.
     */
    start?(): void;
    /**
     * Triggered  **every frame** by the runtime.
     *
     * You should perform your business logic in this method. Example:
     *
     * ```js
     * import { Component, Type } from '@wonderlandengine/api';
     *
     * class TranslateForwardComponent extends Component {
     *     static TypeName = 'translate-forward-component';
     *     static Properties = {
     *         speed: { type: Type.Float, default: 1.0 }
     *     };
     *     constructor() {
     *         this._forward = new Float32Array([0, 0, 0]);
     *     }
     *     update(dt) {
     *         this.object.getForward(this._forward);
     *         this._forward[0] *= this.speed;
     *         this._forward[1] *= this.speed;
     *         this._forward[2] *= this.speed;
     *         this.object.translate(this._forward);
     *     }
     * }
     * ```
     *
     * @param delta Elapsed time between this frame and the previous one, in **seconds**.
     */
    update?(delta: number): void;
    /**
     * Triggered when the component goes from an inactive state to an active state.
     *
     * @note When using ({@link WonderlandEngine.switchTo}), all the components
     * that were previously active will trigger this method.
     *
     * @note You can manually activate or deactivate a component using: {@link Component.active:setter}.
     */
    onActivate?(): void;
    /**
     * Triggered when the component goes from an activated state to an inactive state.
     *
     * @note When using ({@link WonderlandEngine.switchTo}), the components of
     * the scene getting deactivated will trigger this method.
     *
     * @note You can manually activate or deactivate a component using: {@link Component.active:setter}.
     */
    onDeactivate?(): void;
    /**
     * Triggered when the component is removed from its object.
     * For more information, please have a look at {@link Component.onDestroy}.
     *
     * @note This method will not be triggered for inactive scene being destroyed.
     *
     * @since 0.9.0
     */
    onDestroy?(): void;
    /** Manager index. @hidden */
    readonly _manager: number;
    /** Packed id, containing the scene and the local id. @hidden */
    readonly _id: number;
    /** Id relative to the scene component's manager. @hidden */
    readonly _localId: number;
    /**
     * Object containing this object.
     *
     * **Note**: This is cached for faster retrieval.
     *
     * @hidden
     */
    _object: Object3D | null;
    /**
     * Index inside the system, used for ticking.
     *
     * @hidden
     */
    _indexInSystem: number | null;
    /** Scene instance. @hidden */
    protected readonly _scene: Prefab;
    /**
     * Create a new instance
     *
     * @param engine The engine instance.
     * @param manager Index of the manager.
     * @param id WASM component instance index.
     *
     * @hidden
     */
    constructor(scene: Prefab, manager?: number, id?: number);
    /** Scene this component is part of. */
    get scene(): Prefab;
    /** Hosting engine instance. */
    get engine(): WonderlandEngine;
    /** The name of this component's type */
    get type(): string;
    /** The object this component is attached to. */
    get object(): Object3D;
    /**
     * Set whether this component is active.
     *
     * Activating/deactivating a component comes at a small cost of reordering
     * components in the respective component manager. This function therefore
     * is not a trivial assignment.
     *
     * Does nothing if the component is already activated/deactivated.
     *
     * @param active New active state.
     */
    set active(active: boolean);
    /** `true` if the component is marked as active and its scene is active. */
    get active(): boolean;
    /**
     * `true` if the component is marked as active in the scene, `false` otherwise.
     *
     * @note At the opposite of {@link Component.active}, this accessor doesn't
     * take into account whether the scene is active or not.
     */
    get markedActive(): boolean;
    /**
     * Copy all the properties from `src` into this instance.
     *
     * @note Only properties are copied. If a component needs to
     * copy extra data, it needs to override this method.
     *
     * #### Example
     *
     * ```js
     * class MyComponent extends Component {
     *     nonPropertyData = 'Hello World';
     *
     *     copy(src) {
     *         super.copy(src);
     *         this.nonPropertyData = src.nonPropertyData;
     *         return this;
     *     }
     * }
     * ```
     *
     * @note This method is called by {@link Object3D.clone}. Do not attempt to:
     *     - Create new component
     *     - Read references to other objects
     *
     * When cloning via {@link Object3D.clone}, this method will be called before
     * {@link Component.start}.
     *
     * @note JavaScript component properties aren't retargeted. Thus, references
     * inside the source object will not be retargeted to the destination object,
     * at the exception of the skin data on {@link MeshComponent} and {@link AnimationComponent}.
     *
     * @param src The source component to copy from.
     *
     * @returns Reference to self (for method chaining).
     */
    copy(src: Record<string, any>): this;
    /**
     * Remove this component from its objects and destroy it.
     *
     * It is best practice to set the component to `null` after,
     * to ensure it does not get used later.
     *
     * ```js
     *    c.destroy();
     *    c = null;
     * ```
     * @since 0.9.0
     */
    destroy(): void;
    /**
     * Checks equality by comparing ids and **not** the JavaScript reference.
     *
     * @deprecate Use JavaScript reference comparison instead:
     *
     * ```js
     * const componentA = obj.addComponent('mesh');
     * const componentB = obj.addComponent('mesh');
     * const componentC = componentB;
     * console.log(componentA === componentB); // false
     * console.log(componentA === componentA); // true
     * console.log(componentB === componentC); // true
     * ```
     */
    equals(otherComponent: Component | undefined | null): boolean;
    /**
     * Reset the component properties to default.
     *
     * @note This is automatically called during the component instantiation.
     *
     * @returns Reference to self (for method chaining).
     */
    resetProperties(): this;
    /** @deprecated Use {@link Component.resetProperties} instead. */
    reset(): this;
    /**
     * Validate the properties on this instance.
     *
     * @throws If any of the required properties isn't initialized
     * on this instance.
     */
    validateProperties(): void;
    toString(): string;
    /**
     * `true` if the component is destroyed, `false` otherwise.
     *
     * If {@link WonderlandEngine.erasePrototypeOnDestroy} is `true`,
     * reading a custom property will not work:
     *
     * ```js
     * engine.erasePrototypeOnDestroy = true;
     *
     * const comp = obj.addComponent('mesh');
     * comp.customParam = 'Hello World!';
     *
     * console.log(comp.isDestroyed); // Prints `false`
     * comp.destroy();
     * console.log(comp.isDestroyed); // Prints `true`
     * console.log(comp.customParam); // Throws an error
     * ```
     *
     * @since 1.1.1
     */
    get isDestroyed(): boolean;
    /** @hidden */
    _copy(src: this, offsetsPtr: number, copyInfoPtr: number): this;
    /**
     * Trigger the component {@link Component.init} method.
     *
     * @note Use this method instead of directly calling {@link Component.init},
     * because this method creates an handler for the {@link Component.start}.
     *
     * @note This api is meant to be used internally.
     *
     * @hidden
     */
    _triggerInit(): void;
    /**
     * Trigger the component {@link Component.update} method.
     *
     * @note This api is meant to be used internally.
     *
     * @hidden
     */
    _triggerUpdate(dt: number): void;
    /**
     * Trigger the component {@link Component.onActivate} method.
     *
     * @note This api is meant to be used internally.
     *
     * @hidden
     */
    _triggerOnActivate(): void;
    /**
     * Trigger the component {@link Component.onDeactivate} method.
     *
     * @note This api is meant to be used internally.
     *
     * @hidden
     */
    _triggerOnDeactivate(): void;
    /**
     * Trigger the component {@link Component.onDestroy} method.
     *
     * @note This api is meant to be used internally.
     *
     * @hidden
     */
    _triggerOnDestroy(): void;
}
/**
 * Components must be registered before loading / appending a scene.
 *
 * It's possible to end up with a broken component in the following cases:
 *
 * - Component wasn't registered when the scene was loaded
 * - Component instantiation failed
 *
 * This dummy component is thus used as a placeholder by the engine.
 */
export declare class BrokenComponent extends Component {
    static TypeName: string;
}
/**
 * Merge the ascendant properties of class
 *
 * This method walks the prototype chain, and merges
 * all the properties found in parent components.
 *
 * Example:
 *
 * ```js
 * import {Property, inheritProperties} from '@wonderlandengine/api';
 *
 * class Parent {
 *     static Properties = { parentProp: Property.string('parent') };
 * }
 *
 * class Child extends Parent {
 *     static Properties = { childProp: Property.string('child') };
 * }
 * inheritProperties(Child);
 * ```
 *
 * @param target The class in which properties should be merged
 *
 * @hidden
 */
export declare function inheritProperties(target: ComponentConstructor | PropertyRecord): void;
/**
 * Native collision component.
 *
 * Provides access to a native collision component instance.
 */
export declare class CollisionComponent extends Component {
    /** @override */
    static TypeName: string;
    /** @overload */
    getExtents(): Float32Array;
    /**
     * Collision component extents.
     *
     * If {@link collider} returns {@link Collider.Sphere}, only the first
     * component of the returned vector is used.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getExtents<T extends NumberArray>(out: T): T;
    /** Collision component collider */
    get collider(): Collider;
    /**
     * Set collision component collider.
     *
     * @param collider Collider of the collision component.
     */
    set collider(collider: Collider);
    /**
     * Equivalent to {@link CollisionComponent.getExtents}.
     *
     * @note Prefer to use {@link CollisionComponent.getExtents} for performance.
     */
    get extents(): Float32Array;
    /**
     * Set collision component extents.
     *
     * If {@link collider} returns {@link Collider.Sphere}, only the first
     * component of the passed vector is used.
     *
     * Example:
     *
     * ```js
     * // Spans 1 unit on the x-axis, 2 on the y-axis, 3 on the z-axis.
     * collision.extent = [1, 2, 3];
     * ```
     *
     * @param extents Extents of the collision component, expects a
     *      3 component array.
     */
    set extents(extents: Readonly<NumberArray>);
    /**
     * Get collision component radius.
     *
     * @note If {@link collider} is not {@link Collider.Sphere}, the returned value
     * corresponds to the radius of a sphere enclosing the shape.
     *
     * Example:
     *
     * ```js
     * sphere.radius = 3.0;
     * console.log(sphere.radius); // 3.0
     *
     * box.extents = [2.0, 2.0, 2.0];
     * console.log(box.radius); // 1.732...
     * ```
     *
     */
    get radius(): number;
    /**
     * Set collision component radius.
     *
     * @param radius Radius of the collision component
     *
     * @note If {@link collider} is not {@link Collider.Sphere},
     * the extents are set to form a square that fits a sphere with the provided radius.
     *
     * Example:
     *
     * ```js
     * aabbCollision.radius = 2.0; // AABB fits a sphere of radius 2.0
     * boxCollision.radius = 3.0; // Box now fits a sphere of radius 3.0, keeping orientation
     * ```
     *
     */
    set radius(radius: number);
    /**
     * Collision component group.
     *
     * The groups is a bitmask that is compared to other components in {@link CollisionComponent#queryOverlaps}
     * or the group in {@link Scene#rayCast}.
     *
     * Colliders that have no common groups will not overlap with each other. If a collider
     * has none of the groups set for {@link Scene#rayCast}, the ray will not hit it.
     *
     * Each bit represents belonging to a group, see example.
     *
     * ```js
     *    // c belongs to group 2
     *    c.group = (1 << 2);
     *
     *    // c belongs to group 0
     *    c.group = (1 << 0);
     *
     *    // c belongs to group 0 *and* 2
     *    c.group = (1 << 0) | (1 << 2);
     *
     *    (c.group & (1 << 2)) != 0; // true
     *    (c.group & (1 << 7)) != 0; // false
     * ```
     */
    get group(): number;
    /**
     * Set collision component group.
     *
     * @param group Group mask of the collision component.
     */
    set group(group: number);
    /**
     * Set whether this collider is rendered in the debug visualization.
     *
     * @param b Whether this collider is rendered in the debug
     *     visualization.
     * @since 1.5.0
     */
    set visualize(b: boolean);
    /**
     * Whether this collider is rendered in the debug visualization.
     *
     * @since 1.5.0
     */
    get visualize(): boolean;
    /**
     * Query overlapping objects.
     *
     * Usage:
     *
     * ```js
     * const collision = object.getComponent('collision');
     * const overlaps = collision.queryOverlaps();
     * for(const otherCollision of overlaps) {
     *     const otherObject = otherCollision.object;
     *     console.log(`Collision with object ${otherObject.objectId}`);
     * }
     * ```
     *
     * @returns Collision components overlapping this collider.
     */
    queryOverlaps(): CollisionComponent[];
}
/**
 * Native text component
 *
 * Provides access to a native text component instance
 */
export declare class TextComponent extends Component {
    /** @override */
    static TypeName: string;
    /** Text component alignment. */
    get alignment(): Alignment;
    /**
     * Set text component alignment.
     *
     * @param alignment Alignment for the text component.
     */
    set alignment(alignment: Alignment);
    /**
     * Text component vertical alignment.
     * @since 1.2.0
     */
    get verticalAlignment(): VerticalAlignment;
    /**
     * Set text component vertical alignment.
     *
     * @param verticalAlignment Vertical for the text component.
     * @since 1.2.0
     */
    set verticalAlignment(verticalAlignment: VerticalAlignment);
    /**
     * Text component justification.
     *
     * @deprecated Please use {@link TextComponent.verticalAlignment} instead.
     */
    get justification(): VerticalAlignment;
    /**
     * Set text component justification.
     *
     * @param justification Justification for the text component.
     *
     * @deprecated Please use {@link TextComponent.verticalAlignment} instead.
     */
    set justification(justification: VerticalAlignment);
    /**
     * Whether text is justified horizontally.
     *
     * Aligns text horizontally to both margins by adding space between words.
     *
     * Requires {@link wrapMode} to be {@link TextWrapMode.Soft} or
     * {@link TextWrapMode.Hard} and non-0 {@link wrapWidth}.
     *
     * The last line in a paragraph is never justified and respects
     * {@link alignment}.
     *
     * @since 1.3.0
     */
    get justified(): boolean;
    /**
     * Set whether text is justified horizontally.
     *
     * @param justified New justified state for the text component.
     * @since 1.3.0
     */
    set justified(justified: boolean);
    /** Text component character spacing. */
    get characterSpacing(): number;
    /**
     * Set text component character spacing.
     *
     * @param spacing Character spacing for the text component.
     */
    set characterSpacing(spacing: number);
    /** Text component line spacing. */
    get lineSpacing(): number;
    /**
     * Set text component line spacing.
     *
     * @param spacing Line spacing for the text component
     */
    set lineSpacing(spacing: number);
    /** Text component effect. */
    get effect(): TextEffect;
    /**
     * Set text component effect.
     *
     * @param effect Effect for the text component
     */
    set effect(effect: TextEffect);
    /**
     * Equivalent to {@link getEffectOffset}.
     *
     * @note Prefer to use {@link getEffectOffset} for performance.
     *
     * @since 1.3.0
     */
    get effectOffset(): Float32Array;
    /**
     * Equivalent to {@link setEffectOffset}.
     *
     * @since 1.3.0
     */
    set effectOffset(offset: Readonly<NumberArray>);
    /** @overload */
    getEffectOffset(): Float32Array;
    /**
     * Get text component effect offset.
     *
     * @param out Destination array, expected to have at least 2 elements.
     * @returns The `out` parameter.
     *
     * @since 1.3.0
     */
    getEffectOffset<T extends NumberArray>(out: T): T;
    /**
     * Set text component effect offset.
     *
     * The offset is given as X and Y factors for {@link Font.emHeight}. E.g. a
     * value of 2 in one axis offsets the effect by two font line heights. A
     * positive Y value moves the effect **upwards**.
     *
     * @param offset Array with new offset, expected to have at least 2 elements.
     *
     * @since 1.3.0
     */
    setEffectOffset(offset: Readonly<NumberArray>): void;
    /**
     * Text component line wrap mode.
     * @since 1.2.1
     */
    get wrapMode(): TextWrapMode;
    /**
     * Set text component line wrap mode.
     *
     * @param wrapMode Line wrap mode for the text component.
     * @since 1.2.1
     */
    set wrapMode(wrapMode: TextWrapMode);
    /**
     * Text component line wrap width, in object space.
     * @since 1.2.1
     */
    get wrapWidth(): number;
    /**
     * Set text component line wrap width.
     *
     * Only takes effect when {@link wrapMode} is something other than
     * {@link TextWrapMode.None}.
     *
     * @param width Line wrap width for the text component, in object space.
     * @since 1.2.1
     */
    set wrapWidth(width: number);
    /** Text component text. */
    get text(): string;
    /**
     * Set text component text.
     *
     * @param text Text of the text component.
     */
    set text(text: any);
    /**
     * Set material to render the text with.
     *
     * @param material New material.
     */
    set material(material: Material | null | undefined);
    /** Material used to render the text. */
    get material(): Material | null;
    /** @overload */
    getBoundingBoxForText(text: string): Float32Array;
    /**
     * Axis-aligned bounding box for a given text, in object space.
     *
     * To calculate the size for the currently set text, use
     * {@link getBoundingBox}.
     *
     * Useful for calculating the text size before an update and potentially
     * adjusting the text:
     *
     * ```js
     * let updatedName = 'some very long name';
     * const box = new Float32Array(4);
     * text.getBoundingBoxForText(updatedName, box);
     * const width = box[2] - box[0];
     * if(width > 2.0) {
     *     updatedName = updatedName.slice(0, 5) + '...';
     * }
     * text.text = updatedName;
     * ```
     *
     * @param text Text string to calculate the bounding box for.
     * @param out Preallocated array to write into, to avoid garbage,
     *     otherwise will allocate a new Float32Array.
     *
     * @returns Bounding box - left, bottom, right, top.
     */
    getBoundingBoxForText<T extends NumberArray>(text: string, out: T): T;
    /** @overload */
    getBoundingBox(): Float32Array;
    /**
     * Axis-aligned bounding box, in object space.
     *
     * The bounding box is computed using the current component properties
     * that influence the position and size of the text. The bounding box is
     * affected by alignment, spacing, effect type and the font set in the
     * material.
     *
     * To calculate the size for a different text, use
     * {@link getBoundingBoxForText}.
     *
     * Useful for adjusting text position or scaling:
     *
     * ```js
     * const box = new Float32Array(4);
     * text.getBoundingBox(box);
     * const width = box[2] - box[0];
     * // Make text 1m wide
     * text.object.setScalingLocal([1/width, 1, 1]);
     * ```
     *
     * @param text Text string to calculate the bounding box for.
     * @param out Preallocated array to write into, to avoid garbage,
     *     otherwise will allocate a new Float32Array.
     *
     * @returns Bounding box - left, bottom, right, top.
     */
    getBoundingBox<T extends NumberArray>(out: T): T;
}
/**
 * Native view component.
 *
 * Provides access to a native view component instance.
 */
export declare class ViewComponent extends Component {
    /** @override */
    static TypeName: string;
    /**
     * Projection type of the view.
     *
     * @since 1.2.2
     */
    get projectionType(): ProjectionType;
    /**
     * Set the projection type of the view.
     *
     * @param type New projection type.
     * @since 1.2.2
     */
    set projectionType(type: ProjectionType);
    /** @overload */
    getProjectionMatrix(): Float32Array;
    /**
     * Projection matrix.
     *
     * A 4x4 matrix that transforms from view space to a WebGL-compatible clip
     * space (-1 to 1 on all axes, near plane at -1, far plane at 1).
     *
     * If an XR session is active and this is the left or right eye view, this
     * returns the projection matrix reported by the device.
     *
     * @note This is not necessarily the final projection matrix used for
     * rendering. You can use it for unprojecting from screen space coordinates
     * to view space.
     *
     * @param out Destination array/vector, expected to have at least 16 elements.
     * @returns The `out` parameter.
     */
    getProjectionMatrix<T extends NumberArray>(out: T): T;
    /**
     * Equivalent to {@link ViewComponent.getProjectionMatrix}.
     *
     * @note Prefer to use {@link ViewComponent.getProjectionMatrix} for performance.
     */
    get projectionMatrix(): Float32Array;
    /**
     * Override projection matrix for this view.
     *
     * Bypasses the generation of the projection matrix from viewport, fov, near, far.
     *
     * @hidden
     */
    _setProjectionMatrix(v: Readonly<NumberArray>): void;
    /**
     * Generate projection matrix from viewport, fov, near, far.
     *
     * Overwrites any projection matrix set manually.
     *
     * @hidden
     */
    _generateProjectionMatrix(): void;
    /** ViewComponent near clipping plane value. */
    get near(): number;
    /**
     * Set near clipping plane distance for the view.
     *
     * If an XR session is active, the change will apply in the
     * following frame, otherwise the change is immediate.
     *
     * @param near Near depth value.
     */
    set near(near: number);
    /** Far clipping plane value. */
    get far(): number;
    /**
     * Set far clipping plane distance for the view.
     *
     * If an XR session is active, the change will apply in the
     * following frame, otherwise the change is immediate.
     *
     * @param far Near depth value.
     */
    set far(far: number);
    /**
     * Get the horizontal field of view for the view, **in degrees**.
     *
     * If an XR session is active and this is the left or right eye view, this
     * returns the field of view reported by the device, regardless of the fov
     * that was set.
     */
    get fov(): number;
    /**
     * Set the horizontal field of view for the view, **in degrees**.
     *
     * Only has an effect if {@link projectionType} is
     * {@link ProjectionType.Perspective}.
     *
     * If an XR session is active and this is the left or right eye view, the
     * field of view reported by the device is used and this value is ignored.
     * After the XR session ends, the new value is applied.
     *
     * @param fov Horizontal field of view, **in degrees**.
     */
    set fov(fov: number);
    /** @overload */
    getViewport(): Int32Array;
    /**
     * Get viewport of the view of the form: [x, y, width, height].
     *
     * @param out Destination array/vector, expected to have at least 4 elements.
     * @returns The `out` parameter.
     * @since 1.3.0
     */
    getViewport<T extends NumberArray>(out: T): T;
    /**
     * Get viewport of the view of the form: [x, y, width, height].
     * @since 1.3.0
     */
    get viewport(): Int32Array;
    /**
     * Set viewport.
     *
     * @hidden
     */
    _setViewport(x: number, y: number, width: number, height: number): void;
    /**
     * Get the width of the orthographic viewing volume.
     *
     * @since 1.2.2
     */
    get extent(): number;
    /**
     * Set the width of the orthographic viewing volume.
     *
     * Only has an effect if {@link projectionType} is
     * {@link ProjectionType.Orthographic}.
     *
     * @param extent New extent.
     * @since 1.2.2
     */
    set extent(extent: number);
    /**
     * Get the camera exposure multiplier.
     *
     * @since 1.5.2
     */
    get exposure(): number;
    /**
     * Set the camera exposure multiplier.
     *
     * @param exposure New exposure.
     * @since 1.5.2
     */
    set exposure(exposure: number);
    /** @overload */
    getDirectionForScreenPosition(x: number, y: number): Float32Array;
    /**
     * Get the view space direction towards a 2D point in HTML canvas
     * coordinates.
     *
     *  Example:
     *
     * ```js
     * // e holds a MouseEvent, e.g. from a canvas 'click' event listener
     * const direction = new Float32Array(3);
     * viewComponent.getDirectionForScreenPosition(e.clientX, e.clientY, direction);
     * // Get a world space direction
     * viewComponent.object.transformVectorWorld(direction, direction);
     * ```
     *
     * @note For orthographic views, the direction is always (0, 0, -1).
     *
     * @param x X coordinate between 0 and {@link viewport} width (exclusive),
     *     increasing values go right.
     * @param y Y coordinate between 0 and {@link viewport} height (exclusive),
     *     increasing values go **down**.
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     * @since 1.5.0
     */
    getDirectionForScreenPosition<T extends NumberArray>(x: number, y: number, out: T): T;
}
/**
 * Native input component.
 *
 * Provides access to a native input component instance.
 */
export declare class InputComponent extends Component {
    /** @override */
    static TypeName: string;
    /** Input component type */
    get inputType(): InputType;
    /**
     * Set input component type.
     *
     * @params New input component type.
     */
    set inputType(type: InputType);
    /**
     * WebXR Device API input source associated with this input component,
     * if type {@link InputType.ControllerLeft} or {@link InputType.ControllerRight}.
     */
    get xrInputSource(): XRInputSource | null;
    /**
     * 'left', 'right' or `null` depending on the {@link InputComponent#inputType}.
     */
    get handedness(): 'left' | 'right' | null;
}
/**
 * Native light component.
 *
 * Provides access to a native light component instance.
 */
export declare class LightComponent extends Component {
    /** @override */
    static TypeName: string;
    /** @overload */
    getColor(): Float32Array;
    /**
     * Get light color.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     * @since 1.0.0
     */
    getColor<T extends NumberArray>(out: T): T;
    /**
     * Set light color.
     *
     * @param c New color array/vector, expected to have at least 3 elements.
     * @since 1.0.0
     */
    setColor(c: Readonly<NumberArray>): void;
    /**
     * View on the light color.
     *
     * @note Prefer to use {@link getColor} in performance-critical code.
     */
    get color(): Float32Array;
    /**
     * Set light color.
     *
     * @param c Color of the light component.
     *
     * @note Prefer to use {@link setColor} in performance-critical code.
     */
    set color(c: Readonly<NumberArray>);
    /** Light type. */
    get lightType(): LightType;
    /**
     * Set light type.
     *
     * @param lightType Type of the light component.
     */
    set lightType(t: LightType);
    /**
     * Light intensity.
     * @since 1.0.0
     */
    get intensity(): number;
    /**
     * Set light intensity.
     *
     * @param intensity Intensity of the light component.
     * @since 1.0.0
     */
    set intensity(intensity: number);
    /**
     * Outer angle for spot lights, in degrees.
     * @since 1.0.0
     */
    get outerAngle(): number;
    /**
     * Set outer angle for spot lights.
     *
     * @param angle Outer angle, in degrees.
     * @since 1.0.0
     */
    set outerAngle(angle: number);
    /**
     * Inner angle for spot lights, in degrees.
     * @since 1.0.0
     */
    get innerAngle(): number;
    /**
     * Set inner angle for spot lights.
     *
     * @param angle Inner angle, in degrees.
     * @since 1.0.0
     */
    set innerAngle(angle: number);
    /**
     * Whether the light casts shadows.
     * @since 1.0.0
     */
    get shadows(): boolean;
    /**
     * Set whether the light casts shadows.
     *
     * @param b Whether the light casts shadows.
     * @since 1.0.0
     */
    set shadows(b: boolean);
    /**
     * Range for shadows.
     * @since 1.0.0
     */
    get shadowRange(): number;
    /**
     * Set range for shadows.
     *
     * @param range Range for shadows.
     * @since 1.0.0
     */
    set shadowRange(range: number);
    /**
     * Bias value for shadows.
     * @since 1.0.0
     */
    get shadowBias(): number;
    /**
     * Set bias value for shadows.
     *
     * @param bias Bias for shadows.
     * @since 1.0.0
     */
    set shadowBias(bias: number);
    /**
     * Normal bias value for shadows.
     * @since 1.0.0
     */
    get shadowNormalBias(): number;
    /**
     * Set normal bias value for shadows.
     *
     * @param bias Normal bias for shadows.
     * @since 1.0.0
     */
    set shadowNormalBias(bias: number);
    /**
     * Texel size for shadows.
     * @since 1.0.0
     */
    get shadowTexelSize(): number;
    /**
     * Set texel size for shadows.
     *
     * @param size Texel size for shadows.
     * @since 1.0.0
     */
    set shadowTexelSize(size: number);
    /**
     * Cascade count for {@link LightType.Sun} shadows.
     * @since 1.0.0
     */
    get cascadeCount(): number;
    /**
     * Set cascade count for {@link LightType.Sun} shadows.
     *
     * @param count Cascade count.
     * @since 1.0.0
     */
    set cascadeCount(count: number);
}
/**
 * Native animation component.
 *
 * Provides access to a native animation component instance.
 */
export declare class AnimationComponent extends Component {
    /** @override */
    static TypeName: string;
    /**
     * Emitter for animation events triggered on this component.
     *
     * The first argument is the name of the event.
     */
    readonly onEvent: Emitter<[string]>;
    /**
     * Set animation to play.
     *
     * Make sure to {@link Animation#retarget} the animation to affect the
     * right objects.
     *
     * @param anim Animation or animation graph to play.
     */
    set animation(anim: Animation | AnimationGraph | null | undefined);
    /** Animation set for this component */
    get animation(): Animation | null;
    /**
     * Animation graph set for this component
     *
     * @since 1.4.6
     */
    get animationGraph(): AnimationGraph | null;
    /**
     * Set animation graph to play.
     *
     * @param graph Animation graph to play.
     *
     * @since 1.4.6
     */
    set animationGraph(graph: AnimationGraph | null);
    /**
     * Set play count. Set to `0` to loop indefinitely.
     *
     * @param playCount Number of times to repeat the animation.
     */
    set playCount(playCount: number);
    /** Number of times the animation is played. */
    get playCount(): number;
    /**
     * Set speed. Set to negative values to run the animation backwards.
     *
     * Setting speed has an immediate effect for the current frame's update
     * and will continue with the speed from the current point in the animation.
     *
     * @param speed New speed at which to play the animation.
     * @since 0.8.10
     */
    set speed(speed: number);
    /**
     * Speed factor at which the animation is played.
     *
     * @since 0.8.10
     */
    get speed(): number;
    /** Current playing state of the animation */
    get state(): AnimationState;
    /**
     * How to handle root motion on this component.
     *
     * @since 1.2.2
     */
    get rootMotionMode(): RootMotionMode;
    /**
     * Set how to handle root motion.
     *
     * @param mode Mode to handle root motion, see {@link RootMotionMode}.
     * @since 1.2.2
     */
    set rootMotionMode(mode: RootMotionMode);
    /**
     * Current iteration of the animation.
     *
     * If {@link playCount} is not unlimited, the value is in the range from
     * `0` to `playCount`.
     *
     * @since 1.2.3
     */
    get iteration(): number;
    /**
     * Current playing position of the animation within the current iteration,
     * in seconds.
     *
     * The value is in the range from `0.0` to {@link AnimationComponent#duration},
     * if playing in reverse, this range is reversed as well.
     *
     * @since 1.2.3
     */
    get position(): number;
    /**
     * Current duration to loop one iteration in seconds, offers a more accurate duration
     * than {@link Animation#duration} when blending multiple animations.
     *
     * @since 1.2.3
     */
    get duration(): number;
    /**
     * Play animation.
     *
     * If the animation is currently paused, resumes from that position. If the
     * animation is already playing, does nothing.
     *
     * To restart the animation, {@link AnimationComponent#stop} it first.
     */
    play(): void;
    /** Stop animation. */
    stop(): void;
    /** Pause animation. */
    pause(): void;
    /**
     * Get the value of a float parameter in the attached graph.
     * Throws if the parameter is missing.
     *
     * @param name Name of the parameter.
     * @since 1.2.0
     */
    getFloatParameter(name: string): number | null;
    /**
     * Set the value of a float parameter in the attached graph
     * Throws if the parameter is missing.
     *
     * @param name Name of the parameter.
     * @param value Float value to set.
     * @returns 1 if the parameter was successfully set, 0 on fail.
     * @since 1.2.0
     */
    setFloatParameter(name: string, value: number): void;
    /** @overload */
    getRootMotionTranslation(): Float32Array;
    /**
     * Get the root motion translation in **local space** calculated for the current frame.
     *
     * @note If {@link AnimationComponent.rootMotionMode} is not
     * set to {@link RootMotionMode.Script} this will always return an identity translation.
     *
     * @since 1.2.2
     */
    getRootMotionTranslation<T extends NumberArray>(out: T): T;
    /** @overload */
    getRootMotionRotation(): Float32Array;
    /**
     * Get the root motion rotation in **local space** calculated for the current frame.
     *
     * @note If {@link AnimationComponent.rootMotionMode} is not
     * set to {@link RootMotionMode.Script} this will always return an identity rotation.
     *
     * @since 1.2.2
     */
    getRootMotionRotation<T extends NumberArray>(out: T): T;
}
/**
 * Native mesh component.
 *
 * Provides access to a native mesh component instance.
 */
export declare class MeshComponent extends Component {
    /** @override */
    static TypeName: string;
    /**
     * Set material to render the mesh with.
     *
     * @param material Material to render the mesh with.
     */
    set material(material: Material | null | undefined);
    /** Material used to render the mesh. */
    get material(): Material | null;
    /** Mesh rendered by this component. */
    get mesh(): Mesh | null;
    /**
     * Set mesh to rendered with this component.
     *
     * @param mesh Mesh rendered by this component.
     */
    set mesh(mesh: Mesh | null | undefined);
    /** Skin for this mesh component. */
    get skin(): Skin | null;
    /**
     * Set skin to transform this mesh component.
     *
     * @param skin Skin to use for rendering skinned meshes.
     */
    set skin(skin: Skin | null | undefined);
    /**
     * Morph targets for this mesh component.
     *
     * @since 1.2.0
     */
    get morphTargets(): MorphTargets | null;
    /**
     * Set morph targets to transform this mesh component.
     *
     * @param morphTargets Morph targets to use for rendering.
     *
     * @since 1.2.0
     */
    set morphTargets(morphTargets: MorphTargets | null | undefined);
    /**
     * Equivalent to {@link getMorphTargetWeights}.
     *
     * @note Prefer to use {@link getMorphTargetWeights} for performance.
     *
     * @since 1.2.0
     */
    get morphTargetWeights(): Float32Array;
    /**
     * Set the morph target weights to transform this mesh component.
     *
     * @param weights New weights.
     *
     * @since 1.2.0
     */
    set morphTargetWeights(weights: Readonly<NumberArray>);
    /** @overload */
    getMorphTargetWeights(): Float32Array;
    /**
     * Get morph target weights for this mesh component.
     *
     * @param out Destination array, expected to have at least as many elements
     *     as {@link MorphTargets.count}.
     * @returns The `out` parameter.
     *
     * @since 1.2.0
     */
    getMorphTargetWeights<T extends NumberArray>(out: T): T;
    /**
     * Get the weight of a single morph target.
     *
     * @param target Index of the morph target.
     * @returns The weight.
     *
     * @since 1.2.0
     */
    getMorphTargetWeight(target: number): number;
    /**
     * Set morph target weights for this mesh component.
     *
     * @param weights Array of new weights, expected to have at least as many
     *     elements as {@link MorphTargets.count}.
     *
     * @since 1.2.0
     */
    setMorphTargetWeights(weights: Readonly<NumberArray>): void;
    /**
     * Set the weight of a single morph target.
     *
     * @param target Index of the morph target.
     * @param weight The new weight.
     *
     * ## Usage
     *
     * ```js
     * const mesh = object.getComponent('mesh');
     * const mouthTarget = mesh.morphTargets.getTargetIndex('mouth');
     * mesh.setMorphTargetWeight(mouthTarget, 0.5);
     * ```
     *
     * @since 1.2.0
     */
    setMorphTargetWeight(target: number, weight: number): void;
}
/** Replacement policies for spawning particles. */
export declare enum ParticleReplacementPolicy {
    /** Don't replace existing particles. */
    DontReplace = 0,
    /** Replace oldest particles first. */
    ReplaceOldest = 1
}
/**
 * Native particle effect component.
 *
 * @experimental This API might change in upcoming versions.
 *
 * @since 1.5.0
 */
export declare class ParticleEffectComponent extends Component {
    /** @override */
    static TypeName: string;
    /** Particle effect used by this component. */
    get particleEffect(): ParticleEffect | null;
    /**
     * Set particle effect to use with this component.
     *
     * @param particleEffect Particle effect used by this component.
     */
    set particleEffect(particleEffect: ParticleEffect | null | undefined);
    /**
     * Spawn new particles.
     *
     * The spawned particles will always be at the end of the alive range so
     * you can use {@link aliveCount} and the return value to get the index
     * range:
     *
     * ```js
     * const count = comp.spawn(10, 5.0);
     * const end = comp.aliveCount;
     * const start = end - count;
     * for (let i = start; i < end; ++i) {
     *    // Initialize particle i
     * }
     * ```
     *
     * Modifies values in {@link lifetimes}. Manually changing lifetimes will
     * stop {@link spawn} and {@link aliveCount} from working correctly.
     *
     * When particles expire or get replaced, remaining particles will get
     * moved to ensure a contiguous range of alive particles. Use {@link ids}
     * to track individual particles.
     *
     * @param count Number of particles to spawn.
     * @param lifetime Lifetime of the spawned particles, in seconds. If `0`,
     *     the particles don't expire.
     * @param replacementPolicy Policy to handle spawning when the maximum
     *     particle count is reached.
     * @returns Number of spawned particles. Can be less than `count` if the
     *     maximum particle count was reached and the replacement policy is
     *     {@link ParticleReplacementPolicy.DontReplace}.
     */
    spawn(count?: number, lifetime?: number, replacementPolicy?: ParticleReplacementPolicy): number;
    /**
     * Number of particles spawned with {@link spawn} and still alive (i.e.
     * their lifetime hasn't expired yet or was set to `0`).
     *
     * Can be passed to {@link update}.
     */
    get aliveCount(): number;
    /**
     * Get persistent IDs.
     *
     * When using {@link spawn}, particle indices may change as particles
     * expire. You can use IDs as persistent indices into external data or as
     * seeds for generating random data per particle.
     *
     * ID range is between 0 and {@link ParticleEffect.maxCount}. Values are
     * unique for each particle within the component, but not globally.
     *
     * @returns Array of IDs or `null`, if {@link particleEffect} is not set.
     */
    get ids(): Readonly<NumberArray> | null;
    /**
     * Get an attribute accessor to modify particle transforms.
     *
     * Transforms are in local space, analogous to
     * {@link Object3D.setTransformLocal}.
     *
     * @returns Transform accessor or `null`, if {@link particleEffect} is not
     *     set.
     *
     * Call {@link update} for changes to take effect.
     */
    get transforms(): AttributeAccessor<Float32ArrayConstructor> | null;
    /**
     * Get an attribute accessor to modify particle scalings.
     *
     * Scalings are in local space, analogous to
     * {@link Object3D.setScalingLocal}.
     *
     * @returns Scaling accessor or `null`, if {@link particleEffect} is not set.
     *
     * Call {@link update} for changes to take effect.
     */
    get scalings(): AttributeAccessor<Float32ArrayConstructor> | null;
    /**
     * Get an attribute accessor to read or modify particle lifetimes.
     *
     * First component is the elapsed lifetime, second component the total
     * lifetime the particle was created with. Both are in seconds.
     *
     * For convenient lifecycle management, prefer using {@link spawn} and
     * {@link aliveCount} instead of manually modifying lifetimes. Changing
     * lifetimes manually will stop {@link spawn} and {@link aliveCount} from
     * working correctly.
     *
     * @returns Lifetime accessor or `null`, if {@link particleEffect} is not set.
     *
     * Call {@link update} for changes to take effect.
     */
    get lifetimes(): AttributeAccessor<Float32ArrayConstructor> | null;
    /**
     * Get an attribute accessor to modify particle colors.
     *
     * @returns Color accessor or `null`, if {@link particleEffect} is not set.
     *
     * Call {@link update} for changes to take effect.
     */
    get colors(): AttributeAccessor<Float32ArrayConstructor> | null;
    /**
     * Get an attribute accessor to modify particle instance data.
     *
     * @returns Instance data accessor or `null`, if
     *     {@link particleEffect} is not set.
     *
     * Call {@link update} for changes to take effect.
     */
    get instanceData(): AttributeAccessor<Float32ArrayConstructor> | null;
    /**
     * Apply changes to {@link transforms}, {@link scalings},
     * {@link lifetimes}, {@link colors} and {@link instanceData}.
     *
     * Uploads the updated particle attributes to the GPU.
     *
     * Since this is an expensive operation, call it only once you have
     * performed all modifications on a particle effect and avoid calling if
     * you did not perform any modifications at all.
     *
     * @param count Number of particles to update and render, clamped to
     *     {@link ParticleEffect#maxCount}.
     */
    update(count: number): void;
}
/**
 * Enum for Physics axes locking
 *
 * See {@link PhysXComponent.angularLockAxis} and {@link PhysXComponent.linearLockAxis}.
 */
export declare enum LockAxis {
    /**
     * No axis selected.
     */
    None = 0,
    /**
     * **X axis**:
     */
    X = 1,
    /**
     * **Y axis**:
     */
    Y = 2,
    /**
     * **Z axis**:
     */
    Z = 4
}
/**
 * Native physx rigid body component.
 *
 * Provides access to a native mesh component instance.
 * Only available when using physx enabled runtime, see "Project Settings > Runtime".
 */
export declare class PhysXComponent extends Component {
    /** @override */
    static TypeName: string;
    /** @overload */
    getTranslationOffset(): Float32Array;
    /**
     * Local translation offset.
     *
     * Allows to move a physx component without creating a new object in the hierarchy.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     *
     * @since 1.1.1
     */
    getTranslationOffset<T extends NumberArray>(out: T): T;
    /** @overload */
    getRotationOffset(): Float32Array;
    /**
     * Local rotation offset represented as a quaternion.
     *
     * Allows to rotate a physx component without creating a new object in the hierarchy.
     *
     * @param out Destination array/vector, expected to have at least 4 elements.
     * @returns The `out` parameter.
     *
     * @since 1.1.1
     */
    getRotationOffset<T extends NumberArray>(out: T): T;
    /** @overload */
    getExtents(): Float32Array;
    /**
     * The shape extents for collision detection.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getExtents<T extends NumberArray>(out: T): T;
    /** @overload */
    getLinearVelocity(): Float32Array;
    /**
     * Linear velocity or `[0, 0, 0]` if the component is not active.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getLinearVelocity<T extends NumberArray>(out: T): T;
    /** @overload */
    getAngularVelocity(): Float32Array;
    /**
     * Angular velocity or `[0, 0, 0]` if the component is not active.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getAngularVelocity<T extends NumberArray>(out: T): T;
    /**
     * Set whether this rigid body is static.
     *
     * Setting this property only takes effect once the component
     * switches from inactive to active.
     *
     * @param b Whether the rigid body should be static.
     */
    set static(b: boolean);
    /**
     * Whether this rigid body is static.
     *
     * This property returns whether the rigid body is *effectively*
     * static. If static property was set while the rigid body was
     * active, it will not take effect until the rigid body is set
     * inactive and active again. Until the component is set inactive,
     * this getter will return whether the rigid body is actually
     * static.
     */
    get static(): boolean;
    /**
     * Equivalent to {@link PhysXComponent.getTranslationOffset}.
     *
     * Gives a quick view of the offset in a debugger.
     *
     * @note Prefer to use {@link PhysXComponent.getTranslationOffset} for performance.
     *
     * @since 1.1.1
     */
    get translationOffset(): Float32Array;
    /**
     * Set the offset translation.
     *
     * The array must be a vector of at least **3** elements.
     *
     * @note The component must be re-activated to apply the change.
     *
     * @since 1.1.1
     */
    set translationOffset(offset: ArrayLike<number>);
    /**
     * Equivalent to {@link PhysXComponent.getRotationOffset}.
     *
     * Gives a quick view of the offset in a debugger.
     *
     * @note Prefer to use {@link PhysXComponent.getRotationOffset} for performance.
     *
     * @since 1.1.1
     */
    get rotationOffset(): Float32Array;
    /**
     * Set the offset rotation.
     *
     * The array must be a quaternion of at least **4** elements.
     *
     * @note The component must be re-activated to apply the change.
     *
     * @since 1.1.1
     */
    set rotationOffset(offset: ArrayLike<number>);
    /**
     * Set whether this rigid body is kinematic.
     *
     * @param b Whether the rigid body should be kinematic.
     */
    set kinematic(b: boolean);
    /**
     * Whether this rigid body is kinematic.
     */
    get kinematic(): boolean;
    /**
     * Set whether this rigid body's gravity is enabled.
     *
     * @param b Whether the rigid body's gravity should be enabled.
     */
    set gravity(b: boolean);
    /**
     * Whether this rigid body's gravity flag is enabled.
     */
    get gravity(): boolean;
    /**
     * Set whether this rigid body's simulate flag is enabled.
     *
     * @param b Whether the rigid body's simulate flag should be enabled.
     */
    set simulate(b: boolean);
    /**
     * Whether this rigid body's simulate flag is enabled.
     */
    get simulate(): boolean;
    /**
     * Set whether to allow simulation of this rigid body.
     *
     * {@link allowSimulation} and {@link trigger} can not be enabled at the
     * same time. Enabling {@link allowSimulation} while {@link trigger} is enabled
     * will disable {@link trigger}.
     *
     * @param b Whether to allow simulation of this rigid body.
     */
    set allowSimulation(b: boolean);
    /**
     * Whether to allow simulation of this rigid body.
     */
    get allowSimulation(): boolean;
    /**
     * Set whether this rigid body may be queried in ray casts.
     *
     * @param b Whether this rigid body may be queried in ray casts.
     */
    set allowQuery(b: boolean);
    /**
     * Whether this rigid body may be queried in ray casts.
     */
    get allowQuery(): boolean;
    /**
     * Set whether this physics body is a trigger.
     *
     * {@link allowSimulation} and {@link trigger} can not be enabled at the
     * same time. Enabling trigger while {@link allowSimulation} is enabled,
     * will disable {@link allowSimulation}.
     *
     * @param b Whether this physics body is a trigger.
     */
    set trigger(b: boolean);
    /**
     * Whether this physics body is a trigger.
     */
    get trigger(): boolean;
    /**
     * Set whether this physics body is rendered in the debug visualization.
     *
     * @param b Whether this physics body is rendered in the debug
     *     visualization.
     * @since 1.5.0
     */
    set visualize(b: boolean);
    /**
     * Whether this physics body is rendered in the debug visualization.
     *
     * @since 1.5.0
     */
    get visualize(): boolean;
    /**
     * Set the shape for collision detection.
     *
     * @param s New shape.
     * @since 0.8.5
     */
    set shape(s: Shape);
    /** The shape for collision detection. */
    get shape(): Shape;
    /**
     * Set additional data for the shape.
     *
     * Retrieved only from {@link PhysXComponent#shapeData}.
     * @since 0.8.10
     */
    set shapeData(d: {
        index: number;
    } | null);
    /**
     * Additional data for the shape.
     *
     * `null` for {@link Shape} values: `None`, `Sphere`, `Capsule`, `Box`, `Plane`.
     * `{index: n}` for `TriangleMesh` and `ConvexHull`.
     *
     * This data is currently only for passing onto or creating other {@link PhysXComponent}.
     * @since 0.8.10
     */
    get shapeData(): {
        index: number;
    } | null;
    /**
     * Set the shape extents for collision detection.
     *
     * @param e New extents for the shape.
     * @since 0.8.5
     */
    set extents(e: Readonly<NumberArray>);
    /**
     * Equivalent to {@link PhysXComponent.getExtents}.
     *
     * @note Prefer to use {@link PhysXComponent.getExtents} for performance.
     */
    get extents(): Float32Array;
    /**
     * Get staticFriction.
     */
    get staticFriction(): number;
    /**
     * Set staticFriction.
     * @param v New staticFriction.
     */
    set staticFriction(v: number);
    /**
     * Get dynamicFriction.
     */
    get dynamicFriction(): number;
    /**
     * Set dynamicFriction
     * @param v New dynamicDamping.
     */
    set dynamicFriction(v: number);
    /**
     * Get bounciness.
     * @since 0.9.0
     */
    get bounciness(): number;
    /**
     * Set bounciness.
     * @param v New bounciness.
     * @since 0.9.0
     */
    set bounciness(v: number);
    /**
     * Get linearDamping/
     */
    get linearDamping(): number;
    /**
     * Set linearDamping.
     * @param v New linearDamping.
     */
    set linearDamping(v: number);
    /** Get angularDamping. */
    get angularDamping(): number;
    /**
     * Set angularDamping.
     * @param v New angularDamping.
     */
    set angularDamping(v: number);
    /**
     * Set linear velocity.
     *
     * [PhysX Manual - "Velocity"](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/RigidBodyDynamics.html#velocity)
     *
     * Has no effect, if the component is not active.
     *
     * @param v New linear velocity.
     */
    set linearVelocity(v: Readonly<NumberArray>);
    /**
     * Equivalent to {@link PhysXComponent.getLinearVelocity}.
     *
     * @note Prefer to use {@link PhysXComponent.getLinearVelocity} for performance.
     */
    get linearVelocity(): Float32Array;
    /**
     * Set angular velocity
     *
     * [PhysX Manual - "Velocity"](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/RigidBodyDynamics.html#velocity)
     *
     * Has no effect, if the component is not active.
     *
     * @param v New angular velocity
     */
    set angularVelocity(v: Readonly<NumberArray>);
    /**
     * Equivalent to {@link PhysXComponent.getAngularVelocity}.
     *
     * @note Prefer to use {@link PhysXComponent.getAngularVelocity} for performance.
     */
    get angularVelocity(): Float32Array;
    /**
     * Set the components groups mask.
     *
     * @param flags New flags that need to be set.
     */
    set groupsMask(flags: number);
    /**
     * Get the components groups mask flags.
     *
     * Each bit represents membership to group, see example.
     *
     * ```js
     * // Assign c to group 2
     * c.groupsMask = (1 << 2);
     *
     * // Assign c to group 0
     * c.groupsMask  = (1 << 0);
     *
     * // Assign c to group 0 and 2
     * c.groupsMask = (1 << 0) | (1 << 2);
     *
     * (c.groupsMask & (1 << 2)) != 0; // true
     * (c.groupsMask & (1 << 7)) != 0; // false
     * ```
     */
    get groupsMask(): number;
    /**
     * Set the components blocks mask.
     *
     * @param flags New flags that need to be set.
     */
    set blocksMask(flags: number);
    /**
     * Get the components blocks mask flags.
     *
     * Each bit represents membership to the block, see example.
     *
     * ```js
     * // Block overlap with any objects in group 2
     * c.blocksMask = (1 << 2);
     *
     * // Block overlap with any objects in group 0
     * c.blocksMask  = (1 << 0)
     *
     * // Block overlap with any objects in group 0 and 2
     * c.blocksMask = (1 << 0) | (1 << 2);
     *
     * (c.blocksMask & (1 << 2)) != 0; // true
     * (c.blocksMask & (1 << 7)) != 0; // false
     * ```
     */
    get blocksMask(): number;
    /**
     * Set axes to lock for linear velocity.
     *
     * @param lock The Axis that needs to be set.
     *
     * Combine flags with Bitwise OR:
     *
     * ```js
     * body.linearLockAxis = LockAxis.X | LockAxis.Y; // x and y set
     * body.linearLockAxis = LockAxis.X; // y unset
     * ```
     *
     * @note This has no effect if the component is static.
     */
    set linearLockAxis(lock: LockAxis);
    /**
     * Get the linear lock axes flags.
     *
     * To get the state of a specific flag, Bitwise AND with the LockAxis needed.
     *
     * ```js
     * if(body.linearLockAxis & LockAxis.Y) {
     *     console.log("The Y flag was set!");
     * }
     * ```
     *
     * @return axes that are currently locked for linear movement.
     */
    get linearLockAxis(): LockAxis;
    /**
     * Set axes to lock for angular velocity.
     *
     * @param lock The Axis that needs to be set.
     *
     * ```js
     * body.angularLockAxis = LockAxis.X | LockAxis.Y; // x and y set
     * body.angularLockAxis = LockAxis.X; // y unset
     * ```
     *
     * @note This has no effect if the component is static.
     */
    set angularLockAxis(lock: LockAxis);
    /**
     * Get the angular lock axes flags.
     *
     * To get the state of a specific flag, Bitwise AND with the LockAxis needed:
     *
     * ```js
     * if(body.angularLockAxis & LockAxis.Y) {
     *     console.log("The Y flag was set!");
     * }
     * ```
     *
     * @return axes that are currently locked for angular movement.
     */
    get angularLockAxis(): LockAxis;
    /**
     * Set mass.
     *
     * [PhysX Manual - "Mass Properties"](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/RigidBodyDynamics.html#mass-properties)
     *
     * @param m New mass.
     */
    set mass(m: number);
    /** Mass */
    get mass(): number;
    /**
     * Set mass space inertia tensor.
     *
     * [PhysX Manual - "Mass Properties"](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/RigidBodyDynamics.html#mass-properties)
     *
     * Has no effect, if the component is not active.
     *
     * @param v New mass space inertia tensor.
     */
    set massSpaceInertiaTensor(v: Readonly<NumberArray>);
    /**
     * @deprecated Use {@link massSpaceInertiaTensor} instead.
     */
    set massSpaceInteriaTensor(v: Readonly<NumberArray>);
    /**
     * Set the rigid body to sleep upon activation.
     *
     * When asleep, the rigid body will not be simulated until the next contact.
     *
     * @param flag `true` to sleep upon activation.
     *
     * @since 1.1.5
     */
    set sleepOnActivate(flag: boolean);
    /**
     * `true` if the rigid body is set to sleep upon activation, `false` otherwise.
     *
     * @since 1.1.5
     */
    get sleepOnActivate(): boolean;
    /**
     * Apply a force.
     *
     * [PhysX Manual - "Applying Forces and Torques"](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/RigidBodyDynamics.html#applying-forces-and-torques)
     *
     * Has no effect, if the component is not active.
     *
     * @param f Force vector.
     * @param m Force mode, see {@link ForceMode}, default `Force`.
     * @param localForce Whether the force vector is in local space, default `false`.
     * @param p Position to apply force at, default is center of mass.
     * @param local Whether position is in local space, default `false`.
     */
    addForce(f: Readonly<NumberArray>, m?: ForceMode, localForce?: boolean, p?: Readonly<NumberArray>, local?: boolean): void;
    /**
     * Apply torque.
     *
     * [PhysX Manual - "Applying Forces and Torques"](https://gameworksdocs.nvidia.com/PhysX/4.1/documentation/physxguide/Manual/RigidBodyDynamics.html#applying-forces-and-torques)
     *
     * Has no effect, if the component is not active.
     *
     * @param f Force vector.
     * @param m Force mode, see {@link ForceMode}, default `Force`.
     */
    addTorque(f: Readonly<NumberArray>, m?: ForceMode): void;
    /**
     * Add on collision callback.
     *
     * @param callback Function to call when this rigid body (un)collides with any other.
     *
     * ```js
     *  let rigidBody = this.object.getComponent('physx');
     *  rigidBody.onCollision(function(type, other) {
     *      // Ignore uncollides
     *      if(type == CollisionEventType.TouchLost) return;
     *
     *      // Take damage on collision with enemies
     *      if(other.object.name.startsWith("enemy-")) {
     *          this.applyDamage(10);
     *      }
     *  }.bind(this));
     * ```
     *
     * @returns Id of the new callback for use with {@link PhysXComponent#removeCollisionCallback}.
     */
    onCollision(callback: CollisionCallback): number;
    /**
     * Add filtered on collision callback.
     *
     * @param otherComp Component for which callbacks will
     *        be triggered. If you pass this component, the method is equivalent to.
     *        {@link PhysXComponent#onCollision}.
     * @param callback Function to call when this rigid body
     *        (un)collides with `otherComp`.
     * @returns Id of the new callback for use with {@link PhysXComponent#removeCollisionCallback}.
     */
    onCollisionWith(otherComp: this, callback: CollisionCallback): number;
    /**
     * Remove a collision callback added with {@link PhysXComponent#onCollision} or {@link PhysXComponent#onCollisionWith}.
     *
     * @param callbackId Callback id as returned by {@link PhysXComponent#onCollision} or {@link PhysXComponent#onCollisionWith}.
     * @throws When the callback does not belong to the component.
     * @throws When the callback does not exist.
     */
    removeCollisionCallback(callbackId: number): void;
}
/**
 * Access to the physics scene
 */
export declare class Physics {
    /**
     * Hit.
     * @hidden
     */
    _hit: RayHit;
    /**
     * Wonderland Engine instance
     * @hidden
     */
    protected readonly _engine: WonderlandEngine;
    /**
     * Ray Hit
     * @hidden
     */
    private _rayHit;
    constructor(engine: WonderlandEngine);
    /**
     * Cast a ray through the scene and find intersecting physics components.
     *
     * The resulting ray hit will contain **up to 4** closest ray hits,
     * sorted by increasing distance.
     *
     * Example:
     *
     * ```js
     * const hit = engine.physics.rayCast(
     *     [0, 0, 0],
     *     [0, 0, 1],
     *     1 << 0 | 1 << 4, // Only check against physics components in groups 0 and 4
     *     25
     * );
     * if (hit.hitCount > 0) {
     *     const locations = hit.getLocations();
     *     console.log(`Object hit at: ${locations[0][0]}, ${locations[0][1]}, ${locations[0][2]}`);
     * }
     * ```
     *
     * @param o Ray origin.
     * @param d Ray direction.
     * @param groupMask Bitmask of physics groups to filter by: only objects
     *        that are part of given groups are considered for the raycast.
     * @param maxDistance Maximum **inclusive** hit distance. Defaults to `100`.
     *
     * @returns The {@link RayHit} instance, cached by this class.
     *
     * @note The returned {@link RayHit} object is owned by the {@link Physics}
     *       instance and will be reused with the next {@link Physics#rayCast} call.
     */
    rayCast(o: Readonly<NumberArray>, d: Readonly<NumberArray>, groupMask: number, maxDistance?: number): RayHit;
    /** Hosting engine instance. */
    get engine(): WonderlandEngine;
}
/**
 * Mesh index type.
 */
export declare enum MeshIndexType {
    /** Single byte mesh index, range 0-255 */
    UnsignedByte = 1,
    /** Two byte mesh index, range 0-65535 */
    UnsignedShort = 2,
    /** Four byte mesh index, range 0-4294967295 */
    UnsignedInt = 4
}
/**
 * Mesh skinning type.
 */
export declare enum MeshSkinningType {
    /** Not skinned */
    None = 0,
    /** Skinned, 4 joints/weight per vertex */
    FourJoints = 1,
    /** Skinned, 8 joints/weight per vertex */
    EightJoints = 2
}
/**
 * Mesh constructor parameters object.
 *
 * Usage:
 *
 * ```js
 * const mesh = Mesh({vertexCount: 3, indexData: [0, 1, 2]});
 * ```
 */
export interface MeshParameters {
    /** Number of vertices to allocate. */
    vertexCount: number;
    /** Index data values. */
    indexData: Readonly<NumberArray>;
    /** Index type, `null` if not indexed. */
    indexType: MeshIndexType;
    /** Whether the mesh should be skinned. Defaults to not skinned. */
    skinningType: MeshSkinningType;
}
/**
 * Wrapper around a native mesh data.
 *
 * For more information about how to create meshes, have a look at the
 * {@link MeshManager} class.
 *
 * #### Update
 *
 * To modify a mesh, you get access to a {@link AttributeAccessor} that
 * allows you to modify the content of the buffers:
 *
 * Usage:
 *
 * ```js
 * const mesh = engine.es.create({vertexCount: 3, indexData: [0, 1, 2]});
 * const positions = mesh.attribute(MeshAttribute.Position);
 * ...
 * ```
 *
 * @note Meshes are **per-engine**, they can thus be shared by multiple scenes.
 */
export declare class Mesh extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /**
     * @deprecated Use {@link MeshManager.create} instead, accessible via {@link WonderlandEngine.meshes}:
     *
     * ```js
     * const mesh = engine.meshes.create({vertexCount: 3, indexData: [0, 1, 2]});
     * ...
     * mesh.update();
     * ```
     */
    constructor(engine: WonderlandEngine, params: Partial<MeshParameters> | number);
    /** Number of vertices in this mesh. */
    get vertexCount(): number;
    /** Index data (read-only) or `null` if the mesh is not indexed. */
    get indexData(): Uint8Array | Uint16Array | Uint32Array | null;
    /**
     * Apply changes to {@link attribute | vertex attributes}.
     *
     * Uploads the updated vertex attributes to the GPU and updates the bounding
     * sphere to match the new vertex positions.
     *
     * Since this is an expensive operation, call it only once you have performed
     * all modifications on a mesh and avoid calling if you did not perform any
     * modifications at all.
     */
    update(): void;
    /** @overload */
    getBoundingSphere(): Float32Array;
    /** @overload */
    getBoundingSphere<T extends NumberArray>(out: T): T;
    /**
     * Mesh bounding sphere.
     *
     * @param out Preallocated array to write into, to avoid garbage,
     *     otherwise will allocate a new Float32Array.
     *
     * ```js
     *  const sphere = new Float32Array(4);
     *  for(...) {
     *      mesh.getBoundingSphere(sphere);
     *      ...
     *  }
     * ```
     *
     * If the position data is changed, call {@link Mesh#update} to update the
     * bounding sphere.
     *
     * @returns Bounding sphere, 0-2 sphere origin, 3 radius.
     */
    getBoundingSphere<T extends NumberArray>(out: T | Float32Array): T | Float32Array;
    /** @overload */
    attribute(attr: MeshAttribute.Position): AttributeAccessor<Float32ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.Tangent): AttributeAccessor<Float32ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.Normal): AttributeAccessor<Float32ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.TextureCoordinate): AttributeAccessor<Float32ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.Color): AttributeAccessor<Float32ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.JointId): AttributeAccessor<Uint16ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.JointWeight): AttributeAccessor<Float32ArrayConstructor> | null;
    /** @overload */
    attribute(attr: MeshAttribute.SecondaryTextureCoordinate): AttributeAccessor<Float32ArrayConstructor> | null;
    /**
     * Get an attribute accessor to retrieve or modify data of given attribute.
     *
     * @param attr Attribute to get access to
     * @returns Attribute to get access to or `null`, if mesh does not have this attribute.
     *
     * Call {@link update} for changes to vertex attributes to take effect.
     *
     * If there are no shaders in the scene that use `TextureCoordinate` for example,
     * no meshes will have the `TextureCoordinate` attribute.
     *
     * For flexible reusable components, take this into account that only `Position`
     * is guaranteed to be present at all time.
     */
    attribute(attr: MeshAttribute): AttributeAccessor | null;
    /**
     * Destroy and free the meshes memory.
     *
     * It is best practice to set the mesh variable to `null` after calling
     * destroy to prevent accidental use:
     *
     * ```js
     *   mesh.destroy();
     *   mesh = null;
     * ```
     *
     * Accessing the mesh after destruction behaves like accessing an empty
     * mesh.
     *
     * @since 0.9.0
     */
    destroy(): void;
    toString(): string;
}
/**
 * Options to create a new {@link AttributeAccessor} instance.
 */
interface AttributeAccessorOptions<T extends TypedArrayCtor> {
    attribute: number;
    offset: number;
    stride: number;
    formatSize: number;
    componentCount: number;
    arraySize: number;
    length: number;
    bufferType: T;
}
/**
 * An iterator over a mesh vertex attribute.
 *
 * Usage:
 *
 * ```js
 * const mesh = this.object.getComponent('mesh').mesh;
 * const positions = mesh.attribute(MeshAttribute.Position);
 *
 * // Equivalent to `new Float32Array(3)`.
 * const temp = positions.createArray();
 *
 * for(let i = 0; i < positions.length; ++i) {
 *     // `pos` will reference `temp` and thereby not allocate additional
 *     // memory, which would cause a perf spike when collected.
 *     const pos = positions.get(i, temp);
 *     // Scale position by 2 on x-axis only.
 *     pos[0] *= 2.0;
 *     positions.set(i, pos);
 * }
 * // We're done modifying, tell the engine to move vertex data to the GPU.
 * mesh.update();
 * ```
 */
export declare class AttributeAccessor<T extends TypedArrayCtor = TypedArrayCtor> {
    /** Max number of elements. */
    readonly length: number;
    /** Wonderland Engine instance. @hidden */
    protected readonly _engine: WonderlandEngine;
    /** Attribute index. @hidden */
    private _attribute;
    /** Attribute offset. @hidden */
    private _offset;
    /** Attribute stride. @hidden */
    private _stride;
    /** Format size native enum. @hidden */
    private _formatSize;
    /** Number of components per vertex. @hidden */
    private _componentCount;
    /** Number of values per vertex. @hidden */
    private _arraySize;
    /**
     * Class to instantiate an ArrayBuffer to get/set values.
     */
    private _bufferType;
    /**
     * Function to allocate temporary WASM memory. It is cached in the accessor to avoid
     * conditionals during get/set.
     */
    private _tempBufferGetter;
    /**
     * Create a new instance.
     *
     * @note Please use {@link Mesh.attribute} to create a new instance.
     *
     * @param options Contains information about how to read the data.
     * @note Do not use this constructor. Instead, please use the {@link Mesh.attribute} method.
     *
     * @hidden
     */
    constructor(engine: WonderlandEngine, options: AttributeAccessorOptions<T>);
    /**
     * Create a new TypedArray to hold this attribute's values.
     *
     * This method is useful to create a view to hold the data to
     * pass to {@link get} and {@link set}
     *
     * Example:
     *
     * ```js
     * const vertexCount = 4;
     * const positionAttribute = mesh.attribute(MeshAttribute.Position);
     *
     * // A position has 3 floats per vertex. Thus, positions has length 3 * 4.
     * const positions = positionAttribute.createArray(vertexCount);
     * ```
     *
     * @param count The number of **vertices** expected.
     * @returns A TypedArray with the appropriate format to access the data
     */
    createArray(count?: number): TypedArray<T>;
    /** @overload */
    get(index: number): TypedArray<T>;
    /**
     * Get attribute element.
     *
     * @param index Index
     * @param out Preallocated array to write into,
     *      to avoid garbage, otherwise will allocate a new TypedArray.
     *
     * `out.length` needs to be a multiple of the attributes component count, see
     * {@link MeshAttribute}. If `out.length` is more than one multiple, it will be
     * filled with the next n attribute elements, which can reduce overhead
     * of this call.
     *
     * @returns The `out` parameter
     */
    get<T extends NumberArray>(index: number, out: T): T;
    /**
     * Set attribute element.
     *
     * @param i Index
     * @param v Value to set the element to
     *
     * `v.length` needs to be a multiple of the attributes component count, see
     * {@link MeshAttribute}. If `v.length` is more than one multiple, it will be
     * filled with the next n attribute elements, which can reduce overhead
     * of this call.
     *
     * @returns Reference to self (for method chaining)
     */
    set(i: number, v: Readonly<NumberArray>): this;
    /** Hosting engine instance. */
    get engine(): WonderlandEngine;
}
export { AttributeAccessor as MeshAttributeAccessor };
/**
 * Wrapper around a native font resource.
 *
 * @note Fonts are **per-engine**, they can thus be shared by multiple scenes.
 *
 * @since 1.2.0
 */
export declare class Font extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /** Em height in object space. Equivalent to line height. */
    get emHeight(): number;
    /**
     * Cap height in object space. This is the typical height of capital
     * letters. Can be 0 if not defined by the font.
     */
    get capHeight(): number;
    /**
     * X height in object space. This is the typical height of lowercase
     * letters. Can be 0 if not defined by the font.
     */
    get xHeight(): number;
    /**
     * Outline size. This is the factor by which characters are expanded to
     * create the outline effect. Returns 0 if the font was compiled without
     * an outline.
     *
     * @since 1.2.1
     */
    get outlineSize(): number;
}
/**
 * Particle effect constructor parameters object.
 *
 * Usage:
 *
 * ```js
 * const particleEffect = ParticleEffect({maxCount: 10, mesh: myMesh, material: myMaterial});
 * ```
 */
export interface ParticleEffectParameters {
    /** Number of instances to allocate. Can be at most 65535. */
    maxCount: number;
    /** Mesh resource. */
    mesh: Mesh | null;
    /** Material resource. */
    material: Material | null;
    /** Whether to use per-particle colors. */
    colors: boolean;
    /** Whether to use per-particle instance data. */
    instanceData: boolean;
}
/**
 * Wrapper around a particle effect resource.
 *
 * For more information about how to create particle effects, have a look at
 * the {@link ParticleEffectManager} class.
 *
 * @note Particle effects are **per-engine**, they can thus be shared by
 * multiple scenes.
 *
 * @since 1.5.0
 */
export declare class ParticleEffect extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /** Maximum number of particles. */
    get maxCount(): number;
    /**
     * Destroy and free the particle effect memory.
     *
     * It is best practice to set the particle effect variable to `null` after calling
     * destroy to prevent accidental use:
     *
     * ```js
     *   particleEffect.destroy();
     *   particleEffect = null;
     * ```
     *
     * Accessing the particle effect after destruction behaves like accessing
     * an empty particle effect.
     */
    destroy(): void;
    toString(): string;
}
/**
 * Wrapper around a pipeline resource.
 *
 * @note Pipelines are **per-engine**, they can thus be shared by multiple scenes.
 *
 * @since 1.5.1
 */
export declare class Pipeline extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /**
     * Triggered right **before** submitting the draw command.
     *
     * @note {@link Pipeline.renderHooks} must be set to trigger this callback.
     *
     * @experimental This API is **experimental** and might change in upcoming versions.
     *     Use at your own risks.
     *
     * @note Backend should be carefully managed, whether it's WebGL or WebGPU.
     */
    onPreRender: Emitter<void[]>;
    /**
     * Triggered right **after** submitting the draw command.
     *
     * @note {@link Pipeline.renderHooks} must be set to trigger this callback.
     *
     * @experimental This API is **experimental** and might change in upcoming versions.
     *     Use at your own risks.
     *
     * @note Backend should be carefully managed, whether it's WebGL or WebGPU.
     * @note When using the WebGL backend, you will need to re-apply the state exactly as it was.
     */
    onPostRender: Emitter<void[]>;
    /** The pipeline's name */
    get name(): string;
    /**
     * The webgl shader program created for this pipeline
     *
     * @note /!\ Use at your own risk. Backwards compatibility on WebGL state set for the
     *     shaders cannot maintain backwards compatibility and may change at any time.
     *
     * @returns The underlying WebGLProgram or `null` if not using WebGL.
     */
    get webglProgram(): WebGLProgram | null;
    /**
     * Enable / disable render hooks.
     *
     * @note Must be set to `true` for {@link Pipeline.onPreRender} and
     * {@link Pipeline.onPostRender} to be notified.
     */
    set renderHooks(v: boolean);
    /** `true` if render hooks are enabled, `false`, otherwise */
    get renderHooks(): boolean;
}
/**
 * Wrapper around a probe volume scenario resource.
 *
 * @note Probe volume scenarios are **per-engine**, they can thus be shared by
 * multiple scenes.
 *
 * @since 1.5.0
 */
export declare class ProbeVolumeScenario extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    get loaded(): boolean;
}
/**
 * Wrapper around a native texture data.
 *
 * For more information about how to create textures, have a look at the
 * {@link TextureManager} class.
 *
 * @note Textures are **per-engine**, they can thus be shared by multiple scenes.
 */
export declare class Texture extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /**
     * @deprecated Use {@link TextureManager.create} instead, accessible via
     * {@link WonderlandEngine.textures}:
     *
     * ```js
     * const image = new Image();
     * image.onload = () => {
     *     const texture = engine.textures.create(image);
     * };
     * ```
     */
    constructor(engine: WonderlandEngine, param: ImageLike | number);
    /** Whether this texture is valid. */
    get valid(): boolean;
    /**
     * Index in this manager.
     *
     * @deprecated Use {@link Texture.index} instead.
     */
    get id(): number;
    /** Update the texture to match the HTML element (e.g. reflect the current frame of a video). */
    update(): void;
    /**
     * Width of the texture.
     *
     * @note If the actual element's width changed, it must be read from the element itself.
     */
    get width(): number;
    /**
     * Height of the texture.
     *
     * @note If the actual element's height changed, it must be read from the element itself.
     */
    get height(): number;
    /**
     * Returns the html element associated to this texture.
     *
     * @note This accessor will return `null` if the image is compressed.
     */
    get htmlElement(): ImageLike | null;
    /**
     * Upload content to the texture to the GPU.
     *
     * Usage:
     *
     * ```js
     * // Size 128x64
     * const image = new Image();
     * texture.updateSubImage(28, 4, 100, 60, 0, 0, image);
     * ```
     *
     * @note The source is cropped if the destination offset and source dimensions
     * are out-of-bounds regarding the texture size.
     *
     * @note Uploading a texture without a source offset is significantly faster.
     *
     * #### ImageBitmap
     *
     * If you need to upload a subregion, it's recommended to use:
     *
     * ```js
     * createImageBitmap(image, sourceX, sourceY, sourceWidth, sourceHeight).then(bitmap => {
     *     const destinationX = sourceX;
     *     const destinationY = sourceY;
     *     texture.updateSubImage(0, 0, sourceWidth, sourceHeight, destinationX, destinationY, bitmap);
     * });
     * ```
     *
     * You could use this method with a `src` offset, but the synchronous implementation
     * uses a temporary 2d canvas, negatively affecting performance for large content.
     *
     * @param srcX Horizontal pixel offset, in the source image.
     * @param srcY Vertical pixel offset, in the source image.
     * @param srcWidth Width of the sub image, in the source image.
     * @param srcHeight Height of the sub image, in the source image
     * @param dstX Horizontal pixel offset, in the destination image.
     *     Defaults to `srcX` for backward compatibility with older API versions.
     * @param dstY Vertical pixel offset, in the destination image.
     *     Defaults to `srcY` for backward compatibility with older API versions.
     * @param content `Image`, `HTMLCanvasElement`, `HTMLVideoElement`, or `ImageBitmap` to upload.
     * @returns `true` if the update went through, `false` otherwise.
     */
    updateSubImage(srcX: number, srcY: number, srcWidth: number, srcHeight: number, dstX?: number, dstY?: number, content?: ImageLike): boolean;
    /**
     * Destroy and free the texture's texture altas space and memory.
     *
     * It is best practice to set the texture variable to `null` after calling
     * destroy to prevent accidental use of the invalid texture:
     *
     * ```js
     *   texture.destroy();
     *   texture = null;
     * ```
     *
     * @since 0.9.0
     */
    destroy(): void;
    toString(): string;
    private get _imageIndex();
}
/**
 * Wrapper around a native animation.
 */
export declare class Animation extends SceneResource {
    /**
     * @param index Index in the manager
     */
    constructor(host: Prefab | WonderlandEngine | undefined, index: number);
    /** Duration of this animation. */
    get duration(): number;
    /** Number of tracks in this animation. */
    get trackCount(): number;
    /**
     * Clone this animation retargeted to a new set of objects.
     *
     * The clone shares most of the data with the original and is therefore
     * light-weight.
     *
     * **Experimental:** This API might change in upcoming versions.
     *
     * If retargeting to {@link Skin}, the join names will be used to determine a mapping
     * from the previous skin to the new skin. The source skin will be retrieved from
     * the first track in the animation that targets a joint.
     *
     * @param newTargets New targets per track. Expected to have
     *      {@link Animation#trackCount} elements or to be a {@link Skin}.
     * @returns The retargeted clone of this animation.
     */
    retarget(newTargets: Object3D[] | Skin): Animation;
    toString(): string;
}
/**
 * Wrapper around a native animation graph.
 *
 * @since 1.4.6
 */
export declare class AnimationGraph extends SceneResource {
    /**
     * Set the playback speed of a sampler in the graph.
     *
     * @note The index of the sampler is the same as
     * the index of the animation in the graph.
     *
     * #### Usage
     *
     * Following example creates a graph with 1D blending, but reuses
     * the forward walk animation as backwards walk by setting the playbackspeed
     * of its sampler to -1.
     * ```js
     * const graph = engine.animationGraphs.blend1D([walk, idle, walk]);
     * graph.setPlaybackSpeed(0, -1.0);
     * ```
     *
     * @param samplerIndex Index of the sampler
     * @param speed Playback speed to set
     * @since 1.5.0
     * @experimental
     */
    setPlaybackSpeed(samplerIndex: number, speed: number): void;
    toString(): string;
}
/**
 * Scene graph object.
 *
 * Node in the scene graph or "entity". Consists of transformation and a reference
 * to its parent object. Usually holds components and is accessible by components
 * through {@link Component#object}.
 *
 * Objects are stored in a data oriented manner inside WebAssembly memory. This class
 * is a JavaScript API wrapper around this memory for more convenient use in
 * components.
 *
 * Objects can be created and added to a scene through {@link Prefab.addObject}:
 *
 * ```js
 * const parent = scene.addObject();
 * parent.name = 'parent`';
 * const child = scene.addObject(parent);
 * child.name = 'child`';
 * ```
 */
export declare class Object3D {
    /**
     * Packed object id, containing scene index and local id.
     *
     * @hidden
     */
    readonly _id: number;
    /** Object id, relative to the scene manager. @hidden */
    readonly _localId: number;
    /** Scene instance. @hidden */
    protected readonly _scene: Prefab;
    /** Wonderland Engine instance. @hidden */
    protected readonly _engine: WonderlandEngine;
    /**
     * @param o Object id to wrap.
     *
     * @deprecated Objects must be obtained via {@link Scene.addObject} or {@link Scene.wrap}:
     *
     * ```js
     * // Create a new object.
     * const obj = scene.addObject();
     *
     * // Wrap an object using its id. The id must be valid.
     * const obj = scene.wrap(0);
     * ```
     *
     * @hidden
     */
    constructor(scene: WonderlandEngine | Prefab, id: number);
    /**
     * Name of the object.
     *
     * Useful for identifying objects during debugging.
     */
    get name(): string;
    /**
     * Set the object's name.
     *
     * @param newName The new name to set.
     */
    set name(newName: string);
    /**
     * Parent of this object or `null` if parented to root.
     */
    get parent(): Object3D | null;
    /**
     * Equivalent to {@link Object3D.getChildren}.
     *
     * @note Prefer to use {@link Object3D.getChildren} for performance.
     */
    get children(): Object3D[];
    /** The number of children of this object. */
    get childrenCount(): number;
    /**
     * Reparent object to given object.
     *
     * @note Reparenting is not trivial and might have a noticeable performance impact.
     *
     * @param newParent New parent or `null` to parent to root
     */
    set parent(newParent: Object3D | undefined | null);
    /** Local object id in the scene manager. */
    get objectId(): number;
    /** Scene instance. */
    get scene(): Prefab;
    /** Hosting engine instance. */
    get engine(): WonderlandEngine;
    /**
     * Add an object as a child of this instance.
     *
     * @returns A newly created object.
     */
    addChild(): Object3D;
    /**
     * Clone this hierarchy into a new one.
     *
     * Cloning copies the hierarchy structure, object names,
     * as well as components.
     *
     * JavaScript components are cloned using {@link Component.copy}. You can
     * override this method in your components.
     *
     * @param parent The parent for the cloned hierarchy or `null` to clone
     *     into the scene root. Defaults to `null`.
     *
     * @returns The clone of this object.
     */
    clone(parent?: Object3D | null): Object3D;
    /**
     * Children of this object.
     *
     * @note Child order is **undefined**. No assumptions should be made
     * about the index of a specific object.
     *
     * If you need to access a specific child of this object, you can
     * use {@link Object3D.findByName}.
     *
     * When the object exists in the scene at editor time, prefer passing it as
     * a component property.
     *
     * @note When providing an output array, only `this.childrenCount` elements will be written.
     * The rest of the array will not be modified by this method.
     *
     * @param out Destination array, expected to have at least `this.childrenCount` elements.
     * @returns The `out` parameter.
     */
    getChildren(out?: Object3D[]): Object3D[];
    /**
     * Reset local transformation (translation, rotation and scaling) to identity.
     *
     * @returns Reference to self (for method chaining).
     */
    resetTransform(): this;
    /**
     * Reset local position and rotation to identity.
     *
     * @returns Reference to self (for method chaining).
     */
    resetPositionRotation(): this;
    /** @deprecated Please use {@link Object3D.resetPositionRotation} instead. */
    resetTranslationRotation(): this;
    /**
     * Reset local rotation, keep translation.
     *
     * @note To reset both rotation and translation, prefer
     *       {@link resetTranslationRotation}.
     *
     * @returns Reference to self (for method chaining).
     */
    resetRotation(): this;
    /**
     * Reset local translation, keep rotation.
     *
     * @note To reset both rotation and translation, prefer
     *       {@link resetTranslationRotation}.
     *
     * @returns Reference to self (for method chaining).
     */
    resetPosition(): this;
    /** @deprecated Please use {@link Object3D.resetPosition} instead. */
    resetTranslation(): this;
    /**
     * Reset local scaling to identity (``[1.0, 1.0, 1.0]``).
     *
     * @returns Reference to self (for method chaining).
     */
    resetScaling(): this;
    /** @deprecated Please use {@link Object3D.translateLocal} instead. */
    translate(v: Readonly<NumberArray>): this;
    /**
     * Translate object by a vector in the parent's space.
     *
     * @param v Vector to translate by.
     *
     * @returns Reference to self (for method chaining).
     */
    translateLocal(v: Readonly<NumberArray>): this;
    /**
     * Translate object by a vector in object space.
     *
     * @param v Vector to translate by.
     *
     * @returns Reference to self (for method chaining).
     */
    translateObject(v: Readonly<NumberArray>): this;
    /**
     * Translate object by a vector in world space.
     *
     * @param v Vector to translate by.
     *
     * @returns Reference to self (for method chaining).
     */
    translateWorld(v: Readonly<NumberArray>): this;
    /** @deprecated Please use {@link Object3D.rotateAxisAngleDegLocal} instead. */
    rotateAxisAngleDeg(a: Readonly<NumberArray>, d: number): this;
    /**
     * Rotate around given axis by given angle (degrees) in local space.
     *
     * @param a Vector representing the rotation axis.
     * @param d Angle in degrees.
     *
     * @note If the object is translated the rotation will be around
     *     the parent. To rotate around the object origin, use
     *     {@link rotateAxisAngleDegObject}
     *
     * @see {@link rotateAxisAngleRad}
     *
     * @returns Reference to self (for method chaining).
     */
    rotateAxisAngleDegLocal(a: Readonly<NumberArray>, d: number): this;
    /** @deprecated Please use {@link Object3D.rotateAxisAngleRadLocal} instead. */
    rotateAxisAngleRad(a: Readonly<NumberArray>, d: number): this;
    /**
     * Rotate around given axis by given angle (radians) in local space.
     *
     * @param a Vector representing the rotation axis.
     * @param d Angle in radians.
     *
     * @note If the object is translated the rotation will be around
     *     the parent. To rotate around the object origin, use
     *     {@link rotateAxisAngleDegObject}
     *
     * @see {@link rotateAxisAngleDeg}
     *
     * @returns Reference to self (for method chaining).
     */
    rotateAxisAngleRadLocal(a: Readonly<NumberArray>, d: number): this;
    /**
     * Rotate around given axis by given angle (degrees) in object space.
     *
     * @param a Vector representing the rotation axis.
     * @param d Angle in degrees.
     *
     * Equivalent to prepending a rotation quaternion to the object's
     * local transformation.
     *
     * @see {@link rotateAxisAngleRadObject}
     *
     * @returns Reference to self (for method chaining).
     */
    rotateAxisAngleDegObject(a: Readonly<NumberArray>, d: number): this;
    /**
     * Rotate around given axis by given angle (radians) in object space
     * Equivalent to prepending a rotation quaternion to the object's
     * local transformation.
     *
     * @param a Vector representing the rotation axis
     * @param d Angle in degrees
     *
     * @see {@link rotateAxisAngleDegObject}
     *
     * @returns Reference to self (for method chaining).
     */
    rotateAxisAngleRadObject(a: Readonly<NumberArray>, d: number): this;
    /** @deprecated Please use {@link Object3D.rotateLocal} instead. */
    rotate(q: Readonly<NumberArray>): this;
    /**
     * Rotate by a quaternion.
     *
     * @param q the Quaternion to rotate by.
     *
     * @returns Reference to self (for method chaining).
     */
    rotateLocal(q: Readonly<NumberArray>): this;
    /**
     * Rotate by a quaternion in object space.
     *
     * Equivalent to prepending a rotation quaternion to the object's
     * local transformation.
     *
     * @param q the Quaternion to rotate by.
     *
     * @returns Reference to self (for method chaining).
     */
    rotateObject(q: Readonly<NumberArray>): this;
    /** @deprecated Please use {@link Object3D.scaleLocal} instead. */
    scale(v: Readonly<NumberArray>): this;
    /**
     * Scale object by a vector in object space.
     *
     * @param v Vector to scale by.
     *
     * @returns Reference to self (for method chaining).
     */
    scaleLocal(v: Readonly<NumberArray>): this;
    /** @overload */
    getPositionLocal(): Float32Array;
    /**
     * Compute local / object space position from transformation.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getPositionLocal<T extends NumberArray>(out: T): T;
    /** @overload */
    getTranslationLocal(): Float32Array;
    /** @deprecated Please use {@link Object3D.getPositionLocal} instead. */
    getTranslationLocal<T extends NumberArray>(out: T): T;
    /** @overload */
    getPositionWorld(): Float32Array;
    /**
     * Compute world space position from transformation.
     *
     * May recompute transformations of the hierarchy of this object,
     * if they were changed by JavaScript components this frame.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @return The `out` parameter.
     */
    getPositionWorld<T extends NumberArray>(out: T): T;
    /** @overload */
    getTranslationWorld(): Float32Array;
    /** @deprecated Please use {@link Object3D.getPositionWorld} instead. */
    getTranslationWorld<T extends NumberArray>(out: T): T;
    /**
     * Set local / object space position.
     *
     * Concatenates a new translation dual quaternion onto the existing rotation.
     *
     * @param v New local position array/vector, expected to have at least 3 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setPositionLocal(v: Readonly<NumberArray>): this;
    /** @deprecated Please use {@link Object3D.setPositionLocal} instead. */
    setTranslationLocal(v: Readonly<NumberArray>): this;
    /**
     * Set world space position.
     *
     * Applies the inverse parent transform with a new translation dual quaternion
     * which is concatenated onto the existing rotation.
     *
     * @param v New world position array/vector, expected to have at least 3 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setPositionWorld(v: Readonly<NumberArray>): this;
    /** @deprecated Please use {@link Object3D.setPositionWorld} instead. */
    setTranslationWorld(v: Readonly<NumberArray>): this;
    /** @overload */
    getScalingLocal(): Float32Array;
    /**
     * Local / object space scaling.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @return The `out` parameter.
     *
     * @since 1.0.0
     */
    getScalingLocal<T extends NumberArray>(out: T): T;
    /**
     * Set local / object space scaling.
     *
     * @param v New local scaling array/vector, expected to have at least 3 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setScalingLocal(v: Readonly<NumberArray>): this;
    /** @overload */
    getScalingWorld(): Float32Array;
    /**
     * World space scaling.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @return The `out` parameter.
     *
     * @since 1.0.0
     */
    getScalingWorld<T extends NumberArray>(out: T): T;
    /**
     * Set World space scaling.
     *
     * @param v New world scaling array/vector, expected to have at least 3 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setScalingWorld(v: Readonly<NumberArray>): this;
    /** @overload */
    getRotationLocal(): Float32Array;
    /**
     * Local space rotation.
     *
     * @param out Destination array/vector, expected to have at least 4 elements.
     * @return The `out` parameter.
     *
     * @since 1.0.0
     */
    getRotationLocal<T extends NumberArray>(out: T): T;
    /**
     * Set local space rotation.
     *
     * @param v New world rotation array/vector, expected to have at least 4 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setRotationLocal(v: Readonly<NumberArray>): this;
    /** @overload */
    getRotationWorld(): Float32Array;
    /**
     * World space rotation.
     *
     * @param out Destination array/vector, expected to have at least 4 elements.
     * @return The `out` parameter.
     *
     * @since 1.0.0
     */
    getRotationWorld<T extends NumberArray>(out: T): T;
    /**
     * Set local space rotation.
     *
     * @param v New world rotation array/vector, expected to have at least 4 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setRotationWorld(v: Readonly<NumberArray>): this;
    /** @overload */
    getTransformLocal(): Float32Array;
    /**
     * Local space transformation.
     *
     * @param out Destination array/vector, expected to have at least 8 elements.
     * @return The `out` parameter.
     */
    getTransformLocal<T extends NumberArray>(out: T): T;
    /**
     * Set local space rotation.
     *
     * @param v New local transform array, expected to have at least 8 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setTransformLocal(v: Readonly<NumberArray>): this;
    /** @overload */
    getTransformWorld(): Float32Array;
    /**
     * World space transformation.
     *
     * @param out Destination array, expected to have at least 8 elements.
     * @return The `out` parameter.
     */
    getTransformWorld<T extends NumberArray>(out: T): T;
    /**
     * Set world space rotation.
     *
     * @param v New world transform array, expected to have at least 8 elements.
     *
     * @returns Reference to self (for method chaining).
     */
    setTransformWorld(v: Readonly<NumberArray>): this;
    /**
     * Local space transformation.
     *
     * @deprecated Please use {@link Object3D.setTransformLocal} and
     * {@link Object3D.getTransformLocal} instead.
     */
    get transformLocal(): Float32Array;
    /**
     * Set local transform.
     *
     * @param t Local space transformation.
     *
     * @since 0.8.5
     *
     * @deprecated Please use {@link Object3D.setTransformLocal} and
     * {@link Object3D.getTransformLocal} instead.
     */
    set transformLocal(t: Readonly<NumberArray>);
    /**
     * Global / world space transformation.
     *
     * May recompute transformations of the hierarchy of this object,
     * if they were changed by JavaScript components this frame.
     *
     * @deprecated Please use {@link Object3D.setTransformWorld} and
     * {@link Object3D.getTransformWorld} instead.
     */
    get transformWorld(): Float32Array;
    /**
     * Set world transform.
     *
     * @param t Global / world space transformation.
     *
     * @since 0.8.5
     *
     * @deprecated Please use {@link Object3D.setTransformWorld} and
     * {@link Object3D.getTransformWorld} instead.
     */
    set transformWorld(t: Readonly<NumberArray>);
    /**
     * Local / object space scaling.
     *
     * @deprecated Please use {@link Object3D.setScalingLocal} and
     * {@link Object3D.getScalingLocal} instead.
     */
    get scalingLocal(): Float32Array;
    /**
     * Set local space scaling.
     *
     * @param s Local space scaling.
     *
     * @since 0.8.7
     *
     * @deprecated Please use {@link Object3D.setScalingLocal} and
     * {@link Object3D.getScalingLocal} instead.
     */
    set scalingLocal(s: Readonly<NumberArray>);
    /**
     * Global / world space scaling.
     *
     * May recompute transformations of the hierarchy of this object,
     * if they were changed by JavaScript components this frame.
     *
     * @deprecated Please use {@link Object3D.setScalingWorld} and
     * {@link Object3D.getScalingWorld} instead.
     */
    get scalingWorld(): Float32Array;
    /**
     * Set world space scaling.
     *
     * @param t World space scaling.
     *
     * @since 0.8.7
     *
     * @deprecated Please use {@link Object3D.setScalingWorld} and
     * {@link Object3D.getScalingWorld} instead.
     */
    set scalingWorld(s: Readonly<NumberArray>);
    /**
     * Local space rotation.
     *
     * @since 0.8.7
     *
     * @deprecated Please use {@link Object3D.getRotationLocal} and
     * {@link Object3D.setRotationLocal} instead.
     */
    get rotationLocal(): Float32Array;
    /**
     * Global / world space rotation
     *
     * @since 0.8.7
     *
     * @deprecated Please use {@link Object3D.getRotationWorld} and
     * {@link Object3D.setRotationWorld} instead.
     */
    get rotationWorld(): Float32Array;
    /**
     * Set local space rotation.
     *
     * @param r Local space rotation
     *
     * @since 0.8.7
     *
     * @deprecated Please use {@link Object3D.getRotationLocal} and
     * {@link Object3D.setRotationLocal} instead.
     */
    set rotationLocal(r: Readonly<NumberArray>);
    /**
     * Set world space rotation.
     *
     * @param r Global / world space rotation.
     *
     * @since 0.8.7
     *
     * @deprecated Please use {@link Object3D.getRotationWorld} and
     * {@link Object3D.setRotationWorld} instead.
     */
    set rotationWorld(r: Readonly<NumberArray>);
    /** @deprecated Please use {@link Object3D.getForwardWorld} instead. */
    getForward<T extends NumberArray>(out: T): T;
    /**
     * Compute the object's forward facing world space vector.
     *
     * The forward vector in object space is along the negative z-axis, i.e.,
     * `[0, 0, -1]`.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @return The `out` parameter.
     */
    getForwardWorld<T extends NumberArray>(out: T): T;
    /** @deprecated Please use {@link Object3D.getUpWorld} instead. */
    getUp<T extends NumberArray>(out: T): T;
    /**
     * Compute the object's up facing world space vector.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @return The `out` parameter.
     */
    getUpWorld<T extends NumberArray>(out: T): T;
    /** @deprecated Please use {@link Object3D.getRightWorld} instead. */
    getRight<T extends NumberArray>(out: T): T;
    /**
     * Compute the object's right facing world space vector.
     *
     * @param out Destination array/vector, expected to have at least 3 elements.
     * @return The `out` parameter.
     */
    getRightWorld<T extends NumberArray>(out: T): T;
    /**
     * Transform a vector by this object's world transform.
     *
     * @param out Out vector
     * @param v Vector to transform, default `out`
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformVectorWorld<T extends NumberArray>(out: T, v?: NumberArray): T;
    /**
     * Transform a vector by this object's local transform.
     *
     * @param out Out vector
     * @param v Vector to transform, default `out`
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformVectorLocal<T extends NumberArray>(out: T, v?: NumberArray): T;
    /**
     * Transform a point by this object's world transform.
     *
     * @param out Out point.
     * @param p Point to transform, default `out`.
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformPointWorld<T extends NumberArray>(out: T, p?: NumberArray): T;
    /**
     * Transform a point by this object's local transform.
     *
     * @param out Out point.
     * @param p Point to transform, default `out`.
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformPointLocal<T extends NumberArray>(out: T, p?: NumberArray): T;
    /**
     * Transform a vector by this object's inverse world transform.
     *
     * @param out Out vector.
     * @param v Vector to transform, default `out`.
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformVectorInverseWorld<T extends NumberArray>(out: T, v?: NumberArray): T;
    /**
     * Transform a vector by this object's inverse local transform.
     *
     * @param out Out vector
     * @param v Vector to transform, default `out`
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformVectorInverseLocal<T extends NumberArray>(out: T, v?: NumberArray): T;
    /**
     * Transform a point by this object's inverse world transform.
     *
     * @param out Out point.
     * @param p Point to transform, default `out`.
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformPointInverseWorld<T extends NumberArray>(out: T, p?: NumberArray): T;
    /**
     * Transform a point by this object's inverse local transform.
     *
     * @param out Out point.
     * @param p Point to transform, default `out`.
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    transformPointInverseLocal<T extends NumberArray>(out: T, p?: NumberArray): T;
    /**
     * Transform an object space dual quaternion into world space.
     *
     * @param out Out transformation.
     * @param q Local space transformation, default `out`.
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    toWorldSpaceTransform<T extends NumberArray>(out: T, q?: NumberArray): T;
    /**
     * Transform a world space dual quaternion into local space.
     *
     * @param out Out transformation
     * @param q World space transformation, default `out`
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    toLocalSpaceTransform<T extends NumberArray>(out: T, q?: NumberArray): T;
    /**
     * Transform a world space dual quaternion into object space.
     *
     * @param out Out transformation.
     * @param q World space transformation, default `out`
     * @return The `out` parameter.
     *
     * @since 0.8.7
     */
    toObjectSpaceTransform<T extends NumberArray>(out: T, q?: NumberArray): T;
    /**
     * Turn towards / look at target.
     *
     * Rotates the object so that its forward vector faces towards the target
     * position. The `up` vector acts as a hint to uniquely orient the object's
     * up direction. When orienting a view component, the projected `up` vector
     * faces upwards on the viewing plane.
     *
     * @param p Target position to turn towards, in world space.
     * @param up Up vector to align object with, in world space. Default is `[0, 1, 0]`.
     *
     * @returns Reference to self (for method chaining).
     */
    lookAt(p: NumberArray, up?: NumberArray): this;
    /** Destroy the object with all of its components and remove it from the scene */
    destroy(): void;
    /**
     * Mark transformation dirty.
     *
     * Causes an eventual recalculation of {@link transformWorld}, either
     * on next {@link getTranslationWorld}, {@link transformWorld} or
     * {@link scalingWorld} or the beginning of next frame, whichever
     * happens first.
     */
    setDirty(): void;
    /**
     * Disable/enable all components of this object.
     *
     * @param b New state for the components.
     *
     * @since 0.8.5
     */
    set active(b: boolean);
    /** @overload */
    getComponent<K extends keyof NativeComponents>(type: K, index?: number): NativeComponents[K] | null;
    /** @overload */
    getComponent(typeOrClass: string, index?: number): Component | null;
    /**
     * Get a component attached to this object.
     *
     * @param typeOrClass Type name. It's also possible to give a class definition.
     *     In this case, the method will use the `class.TypeName` field to find the component.
     * @param index=0 Index for component of given type. This can be used to access specific
     *      components if the object has multiple components of the same type.
     * @returns The component or `null` if there is no such component on this object
     */
    getComponent<T extends Component>(typeOrClass: ComponentConstructor<T>, index?: number): T | null;
    /** @overload */
    getComponents(): Component[];
    /** @overload */
    getComponents<K extends keyof NativeComponents>(type: K): NativeComponents[K][];
    /** @overload */
    getComponents<T extends Component>(type: string): Component[];
    /**
     * @param typeOrClass Type name, pass a falsey value (`undefined` or `null`) to retrieve all.
     *     It's also possible to give a class definition. In this case, the method will use the `class.TypeName` field to
     *     find the components.
     * @returns All components of given type attached to this object.
     *
     * @note As this function is non-trivial, avoid using it in `update()` repeatedly,
     *      but rather store its result in `init()` or `start()`
     * @warning This method will currently return at most 341 components.
     */
    getComponents<T extends Component>(typeOrClass: ComponentConstructor<T>): T[];
    /** @overload */
    addComponent(type: 'collision', params?: Record<string, any>): CollisionComponent;
    /** @overload */
    addComponent(type: 'text', params?: Record<string, any>): TextComponent;
    /** @overload */
    addComponent(type: 'view', params?: Record<string, any>): ViewComponent;
    /** @overload */
    addComponent(type: 'mesh', params?: Record<string, any>): MeshComponent;
    /** @overload */
    addComponent(type: 'input', params?: Record<string, any>): InputComponent;
    /** @overload */
    addComponent(type: 'light', params?: Record<string, any>): LightComponent;
    /** @overload */
    addComponent(type: 'animation', params?: Record<string, any>): AnimationComponent;
    /** @overload */
    addComponent(type: 'physx', params?: Record<string, any>): PhysXComponent;
    /** @overload */
    addComponent(type: 'particle-effect', params?: Record<string, any>): ParticleEffectComponent;
    /** @overload */
    addComponent(type: string, params?: Record<string, any>): Component;
    /**
     * Add component of given type to the object.
     *
     * You can use this function to clone components, see the example below.
     *
     * ```js
     *  // Clone existing component (since 0.8.10)
     *  let original = this.object.getComponent('mesh');
     *  otherObject.addComponent('mesh', original);
     *  // Create component from parameters
     *  this.object.addComponent('mesh', {
     *      mesh: someMesh,
     *      material: someMaterial,
     *  });
     * ```
     *
     * @param typeOrClass Typename to create a component of. Can be native or
     *     custom JavaScript component type. It's also possible to give a class definition.
     *     In this case, the method will use the `class.TypeName` field.
     * @param params Parameters to initialize properties of the new component,
     *      can be another component to copy properties from.
     *
     * @returns The component or `null` if the type was not found
     */
    addComponent<T extends Component>(typeClass: ComponentConstructor<T>, params?: Record<string, any>): T;
    /**
     * Search for descendants matching the name.
     *
     * This method is a wrapper around {@link Object3D.findByNameDirect} and
     * {@link Object3D.findByNameRecursive}.
     *
     * @param name The name to search for.
     * @param recursive If `true`, the method will look at all the descendants of this object.
     *     If `false`, this method will only perform the search in direct children.
     * @returns An array of {@link Object3D} matching the name.
     *
     * @since 1.1.0
     */
    findByName(name: string, recursive?: boolean): Object3D[];
    /**
     * Search for all **direct** children matching the name.
     *
     * @note Even though this method is heavily optimized, it does perform
     * a linear search to find the objects. Do not use in a hot path.
     *
     * @param name The name to search for.
     * @returns An array of {@link Object3D} matching the name.
     *
     * @since 1.1.0
     */
    findByNameDirect(name: string): Object3D[];
    /**
     * Search for **all descendants** matching the name.
     *
     * @note Even though this method is heavily optimized, it does perform
     * a linear search to find the objects. Do not use in a hot path.
     *
     * @param name The name to search for.
     * @returns An array of {@link Object3D} matching the name.
     *
     * @since 1.1.0
     */
    findByNameRecursive(name: string): Object3D[];
    /**
     * Whether given object's transformation has changed.
     */
    get changed(): boolean;
    /**
     * `true` if the object is destroyed, `false` otherwise.
     *
     * If {@link WonderlandEngine.erasePrototypeOnDestroy} is `true`,
     * reading a custom property will not work:
     *
     * ```js
     * engine.erasePrototypeOnDestroy = true;
     *
     * const obj = scene.addObject();
     * obj.customParam = 'Hello World!';
     *
     * console.log(obj.isDestroyed); // Prints `false`
     * obj.destroy();
     * console.log(obj.isDestroyed); // Prints `true`
     * console.log(obj.customParam); // Throws an error
     * ```
     *
     * @since 1.1.1
     */
    get isDestroyed(): boolean;
    /**
     * `true` if the object is marked as destroyed.
     *
     * This boolean will only ever be `true` when reading objects state
     * from the {@link Component.onDestroy} callback, i.e.,
     *
     * ```js
     * import {Component} from '@wonderlandengine/api';
     * class CleanupComponent extends Component {
     *     onDestroy() {
     *         if (this.object.markedDestroyed) {
     *             // this object is getting removed
     *         } else {
     *             // The component is getting destroyed, the object will
     *             // still exist after the destruction call
     *         }
     *     }
     * }
     * ```
     *
     * Certain operations are forbidden when an object is pending destruction:
     * - Reparenting the object via {@link parent}
     * - {@link addComponent}
     * - {@link Scene.addObject}
     * - {@link Scene.destroy}
     */
    get markedDestroyed(): boolean;
    /**
     * Checks equality by comparing ids and **not** the JavaScript reference.
     *
     * @deprecate Use JavaScript reference comparison instead:
     *
     * ```js
     * const objectA = scene.addObject();
     * const objectB = scene.addObject();
     * const objectC = objectB;
     * console.log(objectA === objectB); // false
     * console.log(objectA === objectA); // true
     * console.log(objectB === objectC); // true
     * ```
     */
    equals(otherObject: Object3D | undefined | null): boolean;
    toString(): string;
}
/**
 * Wrapper around a native skin data.
 */
export declare class Skin extends SceneResource {
    /** Amount of joints in this skin. */
    get jointCount(): number;
    /** Joints object ids for this skin */
    get jointIds(): Uint16Array;
    /**
     * Dual quaternions in a flat array of size 8 times {@link jointCount}.
     *
     * Inverse bind transforms of the skin.
     */
    get inverseBindTransforms(): Float32Array;
    /**
     * Vectors in a flat array of size 3 times {@link jointCount}.
     *
     * Inverse bind scalings of the skin.
     */
    get inverseBindScalings(): Float32Array;
}
/**
 * Wrapper around a native set of morph targets.
 *
 * ## Usage
 *
 * ```js
 * const mesh = object.getComponent('mesh');
 * const mouthTarget = mesh.morphTargets.getTargetIndex('mouth');
 * mesh.setMorphTargetWeight(mouthTarget, 0.5);
 * ```
 *
 * @since 1.2.0
 */
export declare class MorphTargets extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /** Amount of targets in this morph target set. */
    get count(): number;
    /** Returns the name of a given target */
    getTargetName(target: number): string;
    /**
     * Get the index for a given target name.
     *
     * Throws if no target with that name exists.
     *
     * @param name Name of the target.
     */
    getTargetIndex(name: string): number;
}
export { Object3D as Object };
/**
 * Ray hit.
 *
 * Result of a {@link Scene.rayCast} or {@link Physics.rayCast}.
 *
 * @note this class wraps internal engine data and should only be created internally.
 */
export declare class RayHit {
    /** Scene instance. @hidden */
    readonly _scene: Scene;
    /** Pointer to the memory heap. */
    private _ptr;
    /**
     * @param ptr Pointer to the ray hits memory.
     */
    constructor(scene: Scene, ptr: number);
    /** @overload */
    getLocations(): Float32Array[];
    /**
     * Array of ray hit locations.
     *
     * #### Usage
     *
     * ```js
     * const hit = engine.physics.rayCast();
     * if (hit.hitCount > 0) {
     *     const locations = hit.getLocations();
     *     console.log(`Object hit at: ${locations[0][0]}, ${locations[0][1]}, ${locations[0][2]}`);
     * }
     * ```
     *
     * @param out Destination array of arrays/vectors, expected to have at least
     *     `this.hitCount` elements. Each array is expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getLocations<T extends NumberArray[]>(out: T): T;
    /** @overload */
    getNormals(): Float32Array[];
    /**
     * Array of ray hit normals (only when using {@link Physics#rayCast}.
     *
     * @param out Destination array of arrays/vectors, expected to have at least
     *     `this.hitCount` elements. Each array is expected to have at least 3 elements.
     * @returns The `out` parameter.
     */
    getNormals<T extends NumberArray[]>(out: T): T;
    /** @overload */
    getDistances(): Float32Array;
    /**
     * Prefer these to recalculating the distance from locations.
     *
     * Distances of array hits to ray origin.
     *
     * @param out Destination array/vector, expected to have at least this.hitCount elements.
     * @returns The `out` parameter.
     */
    getDistances<T extends NumberArray>(out: T): T;
    /**
     * Array of hit objects.
     *
     * @param out Destination array/vector, expected to have at least `this.hitCount` elements.
     * @returns The `out` parameter.
     */
    getObjects(out?: Object3D[]): Object3D[];
    /** Hosting engine instance. */
    get engine(): WonderlandEngine;
    /**
     * Equivalent to {@link RayHit.getLocations}.
     *
     * @note Prefer to use {@link RayHit.getLocations} for performance.
     */
    get locations(): Float32Array[];
    /**
     * Equivalent to {@link RayHit.getNormals}.
     *
     * @note Prefer to use {@link RayHit.getNormals} for performance.
     */
    get normals(): Float32Array[];
    /**
     * Equivalent to {@link RayHit.getDistances}.
     *
     * @note Prefer to use {@link RayHit.getDistances} for performance.
     */
    get distances(): Float32Array;
    /**
     * Equivalent to {@link RayHit.getObjects}.
     *
     * @note Prefer to use {@link RayHit.getObjects} for performance.
     */
    get objects(): (Object3D | null)[];
    /** Number of hits (max 4) */
    get hitCount(): number;
}
declare class math {
    /** (Experimental!) Cubic Hermite spline interpolation for vector3 and quaternions.
     *
     * With `f == 0`, `out` will be `b`, if `f == 1`, `out` will be c.
     *
     * Whether a quaternion or vector3 interpolation is intended is determined by
     * length of `a`.
     *
     * @param out Array to write result to.
     * @param a First tangent/handle.
     * @param b First point or quaternion.
     * @param c Second point or quaternion.
     * @param d Second handle.
     * @param f Interpolation factor in [0; 1].
     * @returns The `out` parameter.
     *
     * @since 0.8.6
     */
    static cubicHermite<T extends NumberArray>(out: T, a: Readonly<NumberArray>, b: Readonly<NumberArray>, c: Readonly<NumberArray>, d: Readonly<NumberArray>, f: number, engine?: WonderlandEngine): T;
}
export { math };
/**
 * Class for accessing internationalization (i18n) features.
 *
 * Allows {@link I18N.onLanguageChanged "detecting language change"},
 * {@link I18N.language "setting the current language"} or translating
 * {@link I18N.translate "individual terms"}.
 *
 * Internationalization works with terms,
 * a string type keyword that is linked to a different text for each language.
 *
 * Internally, string parameters for text and js components are
 * automatically swapped during language change, given they are linked to a term.
 * If manual text swapping is desired, {@link I18N.translate}
 * can be used to retrieve the current translation for any term.
 *
 * You can also use the {@link I18N.onLanguageChanged} to manually update text
 * when a language is changed to for example update a number in a string.
 *
 * @since 1.0.0
 */
export declare class I18N {
    /**
     * {@link Emitter} for language change events.
     *
     * First parameter to a listener is the old language index,
     * second parameter is the new language index.
     *
     * Usage from a within a component:
     *
     * ```js
     * this.engine.i18n.onLanguageChanged.add((oldLanguageIndex, newLanguageIndex) => {
     *     const oldLanguage = this.engine.i18n.languageName(oldLanguageIndex);
     *     const newLanguage = this.engine.i18n.languageName(newLanguageIndex);
     *     console.log("Switched from", oldLanguage, "to", newLanguage);
     * });
     * ```
     */
    readonly onLanguageChanged: Emitter<[number, number]>;
    /** Wonderland Engine instance. @hidden */
    protected readonly _engine: WonderlandEngine;
    /** Previously set language index. @hidden */
    private _prevLanguageIndex;
    /**
     * Constructor
     */
    constructor(engine: WonderlandEngine);
    /**
     * Set current language and apply translations to linked text parameters.
     *
     * @note This is equivalent to {@link I18N.setLanguage}.
     *
     * @param code Language code to switch to
     */
    set language(code: string | null);
    /** Get current language code. */
    get language(): string | null;
    /**
     * Get the current language index.
     *
     * This method is more efficient than its equivalent:
     *
     * ```js
     * const index = i18n.languageIndex(i18n.language);
     * ```
     */
    get currentIndex(): number;
    /** Previous language index. */
    get previousIndex(): number;
    /**
     * Set current language and apply translations to linked text parameters.
     *
     * @param code The language code.
     * @returns A promise that resolves with the current index code when the
     *     language is loaded.
     */
    setLanguage(code: string | null): Promise<number>;
    /**
     * Get translated string for a term for the currently loaded language.
     *
     * @param term Term to translate
     */
    translate(term: string): string | null;
    /**
     * Get the number of languages in the project.
     *
     */
    languageCount(): number;
    /**
     * Get a language code.
     *
     * @param index Index of the language to get the code from
     */
    languageIndex(code: string): number;
    /**
     * Get a language code.
     *
     * @param index Index of the language to get the code from
     */
    languageCode(index: number): string | null;
    /**
     * Get a language name.
     *
     * @param index Index of the language to get the name from
     */
    languageName(index: number): string | null;
    /** Hosting engine instance. */
    get engine(): WonderlandEngine;
}
/**
 * Environment lighting properties
 *
 * @since 1.2.3
 */
export declare class Environment {
    /** Scene instance. @hidden */
    private readonly _scene;
    /** Constructor */
    constructor(scene: Scene);
    /**
     * Request a re-bake of the environment.
     *
     * @note Currently, baking only takes into account the sky material.
     *
     * @note This API is experimental.
     */
    requestBake(): void;
    /**
     * Get intensity of environment lighting.
     *
     * Incoming environment lighting is multiplied by this factor.
     */
    get intensity(): number;
    /**
     * Set intensity of environment lighting.
     *
     * @param intensity New intensity.
     */
    set intensity(intensity: number);
    /** @overload */
    getTint(): Float32Array;
    /**
     * Get tint for environment lighting.
     *
     * Incoming environment lighting color channels are multiplied by these
     * values.
     *
     * @param out Preallocated array to write into, to avoid garbage,
     *     otherwise will allocate a new Float32Array.
     * @returns Tint values - red, green, blue.
     */
    getTint<T extends NumberArray>(out: T): T;
    /**
     * Equivalent to {@link getTint}.
     *
     * @note Prefer to use {@link getTint} for performance.
     */
    get tint(): Float32Array;
    /**
     * Set tint for environment lighting.
     *
     * @param v New tint value. Expects a 3 component array.
     */
    setTint(v: Readonly<NumberArray>): void;
    /** Equivalent to {@link setTint}. */
    set tint(v: Readonly<NumberArray>);
    /** @overload */
    getCoefficients(): Float32Array;
    /**
     * Get spherical harmonics coefficients for indirect lighting.
     *
     * These are 9 spherical harmonics coefficients for indirect diffuse
     * lighting.
     *
     * @param out Preallocated array to write into, to avoid garbage,
     *     otherwise will allocate a new Float32Array.
     * @returns Spherical harmonics coefficients. Always 27 elements, every
     *     consecutive 3 values representing the red, green, blue components of
     *     a single coefficient. Unused/empty coefficients at the end can be 0.
     */
    getCoefficients<T extends NumberArray>(out: T): T;
    /**
     * Equivalent to {@link getCoefficients}.
     *
     * @note Prefer to use {@link getCoefficients} for performance.
     */
    get coefficients(): Float32Array;
    /**
     * Set spherical harmonics coefficients for indirect lighting.
     *
     * @note The scene must have been packaged with environment lighting on for
     * this to take effect.
     *
     * @example
     *
     * Coefficients from [WebXR Lighting Estimation](https://www.w3.org/TR/webxr-lighting-estimation-1)
     * can be passed as follows:
     *
     * ```js
     * const probe = await engine.xr.session.requestLightProbe();
     * const estimate = engine.xr.frame.getLightEstimate(probe);
     * scene.environment.coefficients = estimate.sphericalHarmonicsCoefficients;
     * ```
     *
     * @param v A set of spherical harmonics coefficients, every 3 elements
     *     constituting the red/green/blue components of a single coefficient.
     *     Should be 0, 3, 12 or 27 **array elements**. Passing an empty array
     *     disables indirect lighting.
     */
    setCoefficients(v: Readonly<NumberArray>): void;
    /** Equivalent to {@link setCoefficients}. */
    set coefficients(v: Readonly<NumberArray>);
}
/**
 * SSAO properties
 *
 * @since 1.6.0
 */
export declare class SSAO {
    /** Scene instance. @hidden */
    private readonly _scene;
    /** Constructor */
    constructor(scene: Scene);
    /** `true` if ssao is enabled, `false` otherwise */
    get enabled(): boolean;
    /**
     * Enable/disable SSAO
     *
     * @note Changing the SSAO state is expensive. To temporarily
     * turn of the effect, set the {@link SSAO.intensity} to 0.0 instead.
     */
    set enabled(value: boolean);
    /** View-space radius. */
    get radius(): number;
    set radius(value: number);
    /**
     * Intensity factor.
     *
     * For physically accurate results, it's recommended to keep
     * the value at `1`.
     */
    get intensity(): number;
    /** View-space radius. */
    set intensity(value: number);
}
/**
 * Wrapper around an audio clip resource.
 *
 * @note Audio clips are **per-engine**, they can thus be shared by
 * multiple scenes.
 *
 * @since 1.5.0
 */
export declare class AudioClip extends Resource {
    /** @hidden */
    static getResourceType(): ResourceType;
    /**
     * Source to the audio clip file.
     *
     * @note Audio clip filenames are generated from their contents hash value,
     * and therefore do not reflect their original namining.
     */
    get source(): string;
}
