import { Version } from './version.js';
import { WonderlandEngine } from './engine.js';
import { ComponentProperty } from './property.js';
import { Logger } from './utils/logger.js';
import { ComponentConstructor, ComponentProto } from './wonderland.js';
import { ImageLike } from './types.js';
/**
 * Low-level wrapper to interact with the WebAssembly code.
 *
 * @hidden
 */
export declare class WASM {
    /**
     * Emscripten wasm field.
     *
     * @note This api is meant to be used internally.
     */
    readonly wasm: ArrayBuffer;
    /**
     * Emscripten canvas.
     *
     * @note This api is meant to be used internally.
     */
    readonly canvas: HTMLCanvasElement;
    /**
     * WebGPU device.
     *
     * @note This api is meant to be used internally.
     */
    readonly preinitializedWebGPUDevice: any;
    /**
     * Convert a WASM memory view to a JavaScript string.
     *
     * @param ptr Pointer start
     * @param ptrEnd Pointer end
     * @returns JavaScript string
     */
    UTF8ViewToString: (ptr: number, ptrEnd: number) => string;
    /** Logger instance. */
    readonly _log: Logger;
    /** If `true`, logs will not spam the console on error. */
    _deactivate_component_on_error: boolean;
    /** Temporary memory pointer. */
    _tempMem: number;
    /** Temporary memory size. */
    _tempMemSize: number;
    /** Temporary float memory view. */
    _tempMemFloat: Float32Array;
    /** Temporary int memory view. */
    _tempMemInt: Int32Array;
    /** Temporary uint8 memory view. */
    _tempMemUint8: Uint8Array;
    /** Temporary uint32 memory view. */
    _tempMemUint32: Uint32Array;
    /** Temporary uint16 memory view. */
    _tempMemUint16: Uint16Array;
    /** Loading screen .bin file data */
    _loadingScreen: ArrayBuffer | null;
    /** List of callbacks triggered when the scene is loaded. */
    readonly _sceneLoadedCallback: any[];
    /** Image cache. */
    _images: (ImageLike | null)[];
    /** Source to load each image in {@link _images}/ */
    _imageSources: (string | Blob | null)[];
    /** Component instances. */
    private _components;
    /** Component Type info. */
    _componentTypes: ComponentConstructor[];
    /** Index per component type name. */
    _componentTypeIndices: Record<string, number>;
    /** Wonderland engine instance. */
    private _engine;
    /**
     * `true` if this runtime is using physx.
     *
     * @note This api is meant to be used internally.
     */
    private _withPhysX;
    /** Decoder for UTF8 `ArrayBuffer` to JavaScript string. */
    private readonly _utf8Decoder;
    /**
     * Registration index of {@link BrokenComponent}.
     *
     * This is used to return dummy instances when a component
     * isn't registered.
     *
     * @hidden
     */
    private readonly _brokenComponentIndex;
    /**
     * Create a new instance of the WebAssembly <> API bridge.
     *
     * @param threads `true` if the runtime used has threads support
     */
    constructor(threads: boolean);
    /** Retrieves the runtime version. */
    get runtimeVersion(): Version;
    /**
     * Reset the cache of the library.
     *
     * @note Should only be called when tearing down the runtime.
     */
    reset(): void;
    /**
     * Checks whether the given component is registered or not.
     *
     * @param ctor  A string representing the component typename (e.g., `'cursor-component'`).
     * @returns `true` if the component is registered, `false` otherwise.
     */
    isRegistered(type: string): boolean;
    /**
     * Register a legacy component in this Emscripten instance.
     *
     * @note This api is meant to be used internally.
     *
     * @param typeName The name of the component.
     * @param params An object containing the parameters (properties).
     * @param object The object's prototype.
     * @returns The registration index
     */
    _registerComponentLegacy(typeName: string, params: Record<string, ComponentProperty>, object: ComponentProto): number;
    /**
     * Register a class component in this Emscripten instance.
     *
     * @note This api is meant to be used internally.
     *
     * @param ctor The class to register.
     * @returns The registration index.
     */
    _registerComponent(ctor: ComponentConstructor): number;
    /**
     * Allocate the requested amount of temporary memory
     * in this WASM instance.
     *
     * @param size The number of bytes to allocate
     */
    allocateTempMemory(size: number): void;
    /**
     * @todo: Delete this and only keep `allocateTempMemory`
     *
     * @param size Number of bytes to allocate
     */
    requireTempMem(size: number): void;
    /**
     * Update the temporary memory views. This must be called whenever the
     * temporary memory address changes.
     *
     * @note This api is meant to be used internally.
     */
    updateTempMemory(): void;
    /**
     * Returns a uint8 buffer view on temporary WASM memory.
     *
     * **Note**: this method might allocate if the requested memory is bigger
     * than the current temporary memory allocated.
     *
     * @param count The number of **elements** required
     * @returns A {@link TypedArray} over the WASM memory
     */
    getTempBufferU8(count: number): Uint8Array;
    /**
     * Returns a uint16 buffer view on temporary WASM memory.
     *
     * **Note**: this method might allocate if the requested memory is bigger
     * than the current temporary memory allocated.
     *
     * @param count The number of **elements** required
     * @returns A {@link TypedArray} over the WASM memory
     */
    getTempBufferU16(count: number): Uint16Array;
    /**
     * Returns a uint32 buffer view on temporary WASM memory.
     *
     * **Note**: this method might allocate if the requested memory is bigger
     * than the current temporary memory allocated.
     *
     * @param count The number of **elements** required.
     * @returns A {@link TypedArray} over the WASM memory.
     */
    getTempBufferU32(count: number): Uint32Array;
    /**
     * Returns a int32 buffer view on temporary WASM memory.
     *
     * **Note**: this method might allocate if the requested memory is bigger
     * than the current temporary memory allocated.
     *
     * @param count The number of **elements** required.
     * @returns A {@link TypedArray} over the WASM memory.
     */
    getTempBufferI32(count: number): Int32Array;
    /**
     * Returns a float32 buffer view on temporary WASM memory.
     *
     * **Note**: this method might allocate if the requested memory is bigger
     * than the current temporary memory allocated.
     *
     * @param count The number of **elements** required.
     * @returns A {@link TypedArray} over the WASM memory.
     */
    getTempBufferF32(count: number): Float32Array;
    /**
     * Copy the string into temporary WASM memory and retrieve the pointer.
     *
     * @note This method will compute the strlen and append a `\0`.
     *
     * @note The result should be used **directly** otherwise it might get
     * overridden by any next call modifying the temporary memory.
     *
     * @param str The string to write to temporary memory
     * @param byteOffset The starting byte offset in the temporary memory at which
     *     the string should be written. This is useful when using multiple temporaries.
     * @return The temporary pointer onto the WASM memory
     */
    tempUTF8(str: string, byteOffset?: number): number;
    /**
     * Copy the buffer into the WASM heap.
     *
     * @note The returned pointer must be freed.
     *
     * @param buffer The buffer to copy into the heap.
     * @returns An allocated pointer, that must be free after use.
     */
    copyBufferToHeap(buffer: ArrayBuffer): number;
    /**
     * Returns `true` if the runtime supports physx or not.
     */
    get withPhysX(): boolean;
    /**
     * Set the engine instance holding this bridge.
     *
     * @note This api is meant to be used internally.
     *
     * @param engine The engine instance.
     */
    protected _setEngine(engine: WonderlandEngine): void;
    protected _wljs_reload(filenamePtr: number): void;
    protected _wljs_init(withPhysX: boolean): void;
    protected _wljs_scene_switch(index: number): void;
    protected _wljs_destroy_image(index: number): void;
    protected _wljs_objects_markDestroyed(sceneIndex: number, idsPtr: number, count: number): void;
    protected _wljs_scene_initialize(sceneIndex: number, idsPtr: number, idsEnd: number, paramDataPtr: number, paramDataEndPtr: number, offsetsPtr: number, offsetsEndPtr: number): void;
    protected _wljs_set_component_param_translation(scene: number, component: number, param: number, valuePtr: number, valueEndPtr: number): void;
    protected _wljs_get_component_type_index(namePtr: number, nameEndPtr: number): number;
    protected _wljs_component_create(sceneIndex: number, index: number, id: number, type: number, object: number): void;
    protected _wljs_component_init(scene: number, component: number): void;
    protected _wljs_component_onActivate(component: number): void;
    protected _wljs_component_onDeactivate(component: number): void;
    protected _wljs_component_markDestroyed(sceneIndex: number, manager: number, componentId: number): void;
    protected _wljs_swap(scene: number, a: number, b: number): void;
    protected _wljs_copy(srcSceneIndex: number, srcIndex: number, dstSceneIndex: number, dstIndex: number, offsetsPtr: number, copyInfoPtr: number): void;
    protected _wljs_scene_postUpdate(dt: number): void;
    /**
     * Forward an animation event to a corresponding
     * {@link AnimationComponent}
     *
     * @note This api is meant to be used internally. Please have a look at
     * {@link AnimationComponent.onEvent} instead.
     *
     * @param componentId Component id in the manager
     * @param namePtr Pointer to UTF8 event name
     * @param nameEndPtr Pointer to end of UTF8 event name
     */
    protected _wljs_trigger_animationEvent(componentId: number, namePtr: number, nameEndPtr: number): void;
    protected _wljs_renderer_onPreRenderDraw(pipelineId: number): void;
    protected _wljs_renderer_onPostRenderDraw(pipelineId: number): void;
}
export interface WASM {
    HEAP8: Int8Array;
    HEAPU8: Uint8Array;
    HEAPU16: Uint16Array;
    HEAP16: Int16Array;
    HEAPU32: Uint32Array;
    HEAP32: Int32Array;
    HEAPF32: Float32Array;
    HEAPF64: Float64Array;
    GL: {
        framebuffers: WebGLFramebuffer[];
        programs: WebGLProgram[];
    };
    assert: (condition: boolean, msg?: string) => void;
    _free: (ptr: number) => void;
    _malloc: (size: number) => number;
    lengthBytesUTF8: (str: string) => number;
    stringToUTF8: (str: string, outPtr: number, len: number) => void;
    UTF8ToString: (ptr: number) => string;
    addFunction: (func: Function, sig: string) => number;
    removeFunction: (ptr: number) => void;
    _wl_runtime_version: (out: number) => void;
    _wl_set_error_callback: (cbPtr: number) => void;
    _wl_set_jobUpdate_callback: (cbPtr: number) => void;
    _wl_application_create: () => void;
    _wl_application_init: (withRenderer: boolean) => boolean;
    _wl_application_exit: () => void;
    _wl_application_destroy: () => void;
    _wl_application_start: () => void;
    _wl_application_stop: () => void;
    _wl_application_redraw: () => void;
    _wl_application_resize: (width: number, height: number) => void;
    _wl_nextUpdate: (delta: number) => void;
    _wl_nextFrame: (delta: number) => void;
    _wl_reset: () => void;
    _wl_reset_context: () => void;
    _wl_xr_init: (viewCount: number, useLayers: boolean, isMultisampled: boolean) => void;
    _wl_xr_focus: () => void;
    _wl_xr_blur: () => void;
    _wl_xr_hide: () => void;
    _wl_xr_exit: () => void;
    _wl_deactivate_activeScene: () => void;
    _wl_renderer_set_mesh_layout: (layout: number) => void;
    _wl_renderer_streaming_reset: () => void;
    _wl_renderer_streaming_idle: () => number;
    _wl_renderer_isReverseZEnabled: () => number;
    _wl_renderer_atlas_downscaleThreshold: () => number;
    _wl_load_main_scene: (ptr: number, size: number, url: number) => number;
    _wl_get_images: (out: number, max: number) => number;
    _wl_get_material_definition_count: () => number;
    _wl_get_material_definition_index: (ptr: number) => number;
    _wl_scene_get_active: (root: number) => number;
    _wl_scene_create: (ptr: number, size: number, url: number) => number;
    _wl_scene_create_chunked_start: (url: number) => number;
    _wl_scene_create_chunked_buffer_size: (index: number) => number;
    _wl_scene_create_chunked_next: (index: number, ptr: number, size: number, readSizePtr: number, requiredSizePtr: number) => boolean;
    _wl_scene_create_chunked_abort: (index: number) => void;
    _wl_scene_create_chunked_end_prefab: (index: number) => number;
    _wl_scene_create_chunked_end_main: (index: number) => void;
    _wl_scene_create_chunked_end_queued: (index: number) => void;
    _wl_scene_create_empty: () => number;
    _wl_scene_initialize: (index: number) => number;
    _wl_scene_destroy: (index: number) => void;
    _wl_scene_instantiate: (src: number, dst: number) => number;
    _wl_scene_activate: (index: number) => void;
    _wl_scene_queued_bin_count: (index: number) => number;
    _wl_scene_queued_bin_path: (sceneIndex: number, index: number) => number;
    _wl_scene_clear_queued_bin_list: (sceneIndex: number) => void;
    _wl_scene_load_queued_bin: (index: number, ptr: number, size: number) => boolean;
    _wl_scene_activatable: (index: number) => boolean;
    _wl_scene_active: (index: number) => boolean;
    _wl_scene_get_baseURL: (index: number) => number;
    _wl_scene_get_filename: (index: number) => number;
    _wl_scene_get_componentsBundle: (index: number) => number;
    _wl_scene_get_component_manager_count: (scene: number) => number;
    _wl_scene_get_component_manager_index: (scene: number, ptr: number) => number;
    _wl_scene_get_mainView: (scene: number) => number;
    _wl_scene_set_mainView: (viewId: number) => number;
    _wl_scene_get_leftView: (scene: number) => number;
    _wl_scene_get_rightView: (scene: number) => number;
    _wl_scene_get_components: (scene: number, typeIndex: number, subTypeIndex: number, active: boolean, offset: number, count: number, ptr: number) => number;
    _wl_scene_get_component: (scene: number, typeIndex: number, index: number) => number;
    _wl_scene_ray_cast: (scene: number, x: number, y: number, z: number, dx: number, dy: number, dz: number, groupMask: number, maxDistance: number, outPtr: number) => void;
    _wl_scene_add_object: (scene: number, parentId: number) => number;
    _wl_scene_add_objects: (scene: number, parentId: number, count: number, componentCountHint: number, ptr: number, size: number) => number;
    _wl_scene_reserve_objects: (scene: number, objectCount: number, _tempMem: number) => void;
    _wl_scene_set_sky_material: (index: number, id: number) => void;
    _wl_scene_get_sky_material: (index: number) => number;
    _wl_scene_environment_bake: (index: number) => number;
    _wl_scene_environment_set_intensity: (index: number, intensity: number) => void;
    _wl_scene_environment_get_intensity: (index: number) => number;
    _wl_scene_environment_set_tint: (index: number, r: number, g: number, b: number) => void;
    _wl_scene_environment_get_tint: (index: number, ptr: number) => void;
    _wl_scene_environment_set_coefficients: (index: number, ptr: number, count: number) => void;
    _wl_scene_environment_get_coefficients: (index: number, ptr: number) => void;
    _wl_scene_ssao_get_enabled: (index: number) => boolean;
    _wl_scene_ssao_set_enabled: (index: number, value: boolean) => void;
    _wl_scene_ssao_get_radius: (index: number) => number;
    _wl_scene_ssao_set_radius: (index: number, value: number) => void;
    _wl_scene_ssao_get_intensity: (index: number) => number;
    _wl_scene_ssao_set_intensity: (index: number, value: number) => void;
    _wl_scene_set_clearColor: (r: number, g: number, b: number, a: number) => void;
    _wl_scene_enableColorClear: (b: boolean) => void;
    _wl_scene_set_probeVolumeScenario: (index: number, scenarioId: number) => void;
    _wl_scene_get_probeVolumeScenario: (index: number) => number;
    _wl_set_loading_screen_progress: (ratio: number) => void;
    _wl_glTF_scene_create: (extensions: boolean, generateSimpleTangents: boolean, async: boolean, ptr: number, ptrEnd: number) => number;
    _wl_glTF_scene_get_extensions: (index: number) => number;
    _wl_glTF_scene_extensions_gltfIndex_to_id: (gltfScene: number, destScene: number, objectIndex: number, gltfIndex: number) => number;
    _wl_component_get_object: (manager: number, id: number) => number;
    _wl_component_setActive: (manager: number, id: number, active: boolean) => void;
    _wl_component_isActive: (manager: number, id: number) => number;
    _wl_component_remove: (manager: number, id: number) => void;
    _wl_collision_component_get_collider: (id: number) => number;
    _wl_collision_component_set_collider: (id: number, collider: number) => void;
    _wl_collision_component_get_extents: (id: number) => number;
    _wl_collision_component_get_group: (id: number) => number;
    _wl_collision_component_set_group: (id: number, group: number) => void;
    _wl_collision_component_get_visualize: (id: number) => number;
    _wl_collision_component_set_visualize: (id: number, visualize: boolean) => void;
    _wl_collision_component_query_overlaps: (id: number, outPtr: number, outCount: number) => number;
    _wl_text_component_get_horizontal_alignment: (id: number) => number;
    _wl_text_component_set_horizontal_alignment: (id: number, alignment: number) => void;
    _wl_text_component_get_vertical_alignment: (id: number) => number;
    _wl_text_component_set_vertical_alignment: (id: number, verticalAlignment: number) => void;
    _wl_text_component_get_justified: (id: number) => number;
    _wl_text_component_set_justified: (id: number, justified: boolean) => void;
    _wl_text_component_get_character_spacing: (id: number) => number;
    _wl_text_component_set_character_spacing: (id: number, spacing: number) => void;
    _wl_text_component_get_line_spacing: (id: number) => number;
    _wl_text_component_set_line_spacing: (id: number, spacing: number) => void;
    _wl_text_component_get_effect: (id: number) => number;
    _wl_text_component_get_effectOffset: (id: number, outPtr: number) => void;
    _wl_text_component_set_effect: (id: number, effect: number) => void;
    _wl_text_component_set_effectOffset: (id: number, offsetPtr: number) => void;
    _wl_text_component_get_wrapMode: (id: number) => number;
    _wl_text_component_set_wrapMode: (id: number, mode: number) => void;
    _wl_text_component_get_wrapWidth: (id: number) => number;
    _wl_text_component_set_wrapWidth: (id: number, width: number) => void;
    _wl_text_component_get_text: (id: number) => number;
    _wl_text_component_set_text: (id: number, ptr: number) => void;
    _wl_text_component_set_material: (id: number, materialId: number) => void;
    _wl_text_component_get_material: (id: number) => number;
    _wl_text_component_get_boundingBox: (id: number, textPtr: number, resultPtr: number) => number;
    _wl_view_component_get_projectionType: (id: number) => number;
    _wl_view_component_set_projectionType: (id: number, type: number) => void;
    _wl_view_component_get_projectionMatrix: (id: number, ptr: number) => void;
    _wl_view_component_set_projectionMatrix: (id: number, ptr: number) => void;
    _wl_view_component_remapProjectionMatrix: (id: number, inverseDepth: boolean, depthZeroToOne: boolean) => void;
    _wl_view_component_generate_projectionMatrix: (id: number) => void;
    _wl_view_component_get_near: (id: number) => number;
    _wl_view_component_set_near: (id: number, near: number) => void;
    _wl_view_component_get_far: (id: number) => number;
    _wl_view_component_set_far: (id: number, far: number) => void;
    _wl_view_component_get_fov: (id: number) => number;
    _wl_view_component_set_fov: (id: number, fov: number) => void;
    _wl_view_component_get_viewport: (id: number) => number;
    _wl_view_component_set_viewport: (id: number, x: number, y: number, width: number, height: number) => void;
    _wl_view_component_get_extent: (id: number) => number;
    _wl_view_component_set_extent: (id: number, fov: number) => void;
    _wl_view_component_get_exposure: (id: number) => number;
    _wl_view_component_set_exposure: (id: number, exposure: number) => void;
    _wl_view_component_set_externalFramebuffer: (id: number, fb: number) => void;
    _wl_view_component_get_directionForScreenPosition: (id: number, x: number, y: number, ptr: number) => void;
    _wl_input_component_get_type: (id: number) => number;
    _wl_input_component_set_type: (id: number, type: number) => void;
    _wl_input_set_transformation: (sceneIndex: number, type: number, ptr: number) => void;
    _wl_light_component_get_color: (id: number) => number;
    _wl_light_component_get_type: (id: number) => number;
    _wl_light_component_set_type: (id: number, type: number) => void;
    _wl_light_component_get_intensity: (id: number) => number;
    _wl_light_component_set_intensity: (id: number, intensity: number) => void;
    _wl_light_component_get_outerAngle: (id: number) => number;
    _wl_light_component_set_outerAngle: (id: number, angle: number) => void;
    _wl_light_component_get_innerAngle: (id: number) => number;
    _wl_light_component_set_innerAngle: (id: number, angle: number) => void;
    _wl_light_component_get_shadows: (id: number) => number;
    _wl_light_component_set_shadows: (id: number, shadows: boolean) => void;
    _wl_light_component_get_shadowRange: (id: number) => number;
    _wl_light_component_set_shadowRange: (id: number, range: number) => void;
    _wl_light_component_get_shadowBias: (id: number) => number;
    _wl_light_component_set_shadowBias: (id: number, bias: number) => void;
    _wl_light_component_get_shadowNormalBias: (id: number) => number;
    _wl_light_component_set_shadowNormalBias: (id: number, bias: number) => void;
    _wl_light_component_get_shadowTexelSize: (id: number) => number;
    _wl_light_component_set_shadowTexelSize: (id: number, size: number) => void;
    _wl_light_component_get_cascadeCount: (id: number) => number;
    _wl_light_component_set_cascadeCount: (id: number, count: number) => void;
    _wl_animation_component_get_animation: (id: number) => number;
    _wl_animation_component_get_animationGraph: (id: number) => number;
    _wl_animation_component_set_animation: (id: number, animId: number) => void;
    _wl_animation_component_set_animationGraph: (id: number, animGraphId: number) => void;
    _wl_animation_component_get_playCount: (id: number) => number;
    _wl_animation_component_set_playCount: (id: number, count: number) => void;
    _wl_animation_component_get_speed: (id: number) => number;
    _wl_animation_component_set_speed: (id: number, speed: number) => void;
    _wl_animation_component_play: (id: number) => void;
    _wl_animation_component_stop: (id: number) => void;
    _wl_animation_component_pause: (id: number) => void;
    _wl_animation_component_state: (id: number) => number;
    _wl_animation_component_getGraphParamValue: (id: number, paramIndex: number, outPtr: number) => number;
    _wl_animation_component_setGraphParamValue: (id: number, paramIndex: number, valuePtr: number) => void;
    _wl_animation_component_get_rootMotionMode: (id: number) => number;
    _wl_animation_component_set_rootMotionMode: (id: number, value: number) => void;
    _wl_animation_component_get_rootMotion_translation: (id: number, outPtr: number) => number;
    _wl_animation_component_get_rootMotion_rotation: (id: number, outPtr: number) => number;
    _wl_animation_component_getGraphParamIndex: (id: number, paramName: number) => number;
    _wl_animation_component_get_iteration: (id: number) => number;
    _wl_animation_component_get_position: (id: number) => number;
    _wl_animation_component_get_duration: (id: number) => number;
    _wl_mesh_component_get_material: (id: number) => number;
    _wl_mesh_component_set_material: (id: number, materialId: number) => void;
    _wl_mesh_component_get_mesh: (id: number) => number;
    _wl_mesh_component_set_mesh: (id: number, meshId: number) => void;
    _wl_mesh_component_get_skin: (id: number) => number;
    _wl_mesh_component_set_skin: (id: number, skinId: number) => void;
    _wl_mesh_component_get_morph_targets: (id: number) => number;
    _wl_mesh_component_set_morph_targets: (id: number, morphTargetSetId: number) => void;
    _wl_mesh_component_get_morph_target_weight: (id: number, index: number) => number;
    _wl_mesh_component_get_morph_target_weights: (id: number, ptr: number) => number;
    _wl_mesh_component_set_morph_target_weight: (id: number, index: number, weight: number) => void;
    _wl_mesh_component_set_morph_target_weights: (id: number, ptr: number, count: number) => void;
    _wl_particleEffect_component_get_particleEffect: (id: number) => number;
    _wl_particleEffect_component_set_particleEffect: (id: number, particleEffectId: number) => void;
    _wl_particleEffect_component_get_ids: (id: number) => number;
    _wl_particleEffect_component_get_transforms: (id: number) => number;
    _wl_particleEffect_component_get_scalings: (id: number) => number;
    _wl_particleEffect_component_get_lifetimes: (id: number) => number;
    _wl_particleEffect_component_get_colors: (id: number) => number;
    _wl_particleEffect_component_get_instanceData: (id: number) => number;
    _wl_particleEffect_component_update: (id: number, count: number) => void;
    _wl_particleEffect_component_spawnParticles: (id: number, count: number, lifetime: number, replacementPolicy: number) => number;
    _wl_particleEffect_component_aliveCount: (id: number) => number;
    _wl_physx_component_get_static: (id: number) => number;
    _wl_physx_component_set_static: (id: number, flag: boolean) => void;
    _wl_physx_component_get_kinematic: (id: number) => number;
    _wl_physx_component_set_kinematic: (id: number, kinematic: boolean) => void;
    _wl_physx_component_get_gravity: (id: number) => number;
    _wl_physx_component_set_gravity: (id: number, gravity: boolean) => void;
    _wl_physx_component_get_simulate: (id: number) => number;
    _wl_physx_component_set_simulate: (id: number, simulation: boolean) => void;
    _wl_physx_component_get_allowSimulation: (id: number) => number;
    _wl_physx_component_set_allowSimulation: (id: number, allowSimulation: boolean) => void;
    _wl_physx_component_get_allowQuery: (id: number) => number;
    _wl_physx_component_set_allowQuery: (id: number, allowQuery: boolean) => void;
    _wl_physx_component_get_trigger: (id: number) => number;
    _wl_physx_component_set_trigger: (id: number, trigger: boolean) => void;
    _wl_physx_component_get_visualize: (id: number) => number;
    _wl_physx_component_set_visualize: (id: number, visualize: boolean) => void;
    _wl_physx_component_get_shape: (id: number) => number;
    _wl_physx_component_set_shape: (id: number, shape: number) => void;
    _wl_physx_component_get_shape_data: (id: number) => number;
    _wl_physx_component_set_shape_data: (id: number, shapeIndex: number) => void;
    _wl_physx_component_get_extents: (id: number) => number;
    _wl_physx_component_get_staticFriction: (id: number) => number;
    _wl_physx_component_set_staticFriction: (id: number, value: number) => void;
    _wl_physx_component_get_dynamicFriction: (id: number) => number;
    _wl_physx_component_set_dynamicFriction: (id: number, value: number) => void;
    _wl_physx_component_get_bounciness: (id: number) => number;
    _wl_physx_component_set_bounciness: (id: number, value: number) => void;
    _wl_physx_component_get_linearDamping: (id: number) => number;
    _wl_physx_component_set_linearDamping: (id: number, value: number) => void;
    _wl_physx_component_get_angularDamping: (id: number) => number;
    _wl_physx_component_set_angularDamping: (id: number, value: number) => void;
    _wl_physx_component_get_linearVelocity: (id: number, ptr: number) => number;
    _wl_physx_component_set_linearVelocity: (id: number, x: number, y: number, z: number) => void;
    _wl_physx_component_get_angularVelocity: (id: number, ptr: number) => number;
    _wl_physx_component_set_angularVelocity: (id: number, x: number, y: number, z: number) => void;
    _wl_physx_component_get_groupsMask: (id: number) => number;
    _wl_physx_component_set_groupsMask: (id: number, flags: number) => void;
    _wl_physx_component_get_blocksMask: (id: number) => number;
    _wl_physx_component_set_blocksMask: (id: number, flags: number) => void;
    _wl_physx_component_get_linearLockAxis: (id: number) => number;
    _wl_physx_component_set_linearLockAxis: (id: number, lock: number) => void;
    _wl_physx_component_get_angularLockAxis: (id: number) => number;
    _wl_physx_component_set_angularLockAxis: (id: number, lock: number) => void;
    _wl_physx_component_get_mass: (id: number) => number;
    _wl_physx_component_set_mass: (id: number, value: number) => void;
    _wl_physx_component_get_offsetTranslation: (id: number, out: number) => void;
    _wl_physx_component_get_offsetTransform: (id: number) => number;
    _wl_physx_component_set_offsetTranslation: (id: number, x: number, y: number, z: number) => void;
    _wl_physx_component_set_offsetRotation: (id: number, x: number, y: number, z: number, w: number) => void;
    _wl_physx_component_set_sleepOnActivate: (id: number, flag: boolean) => void;
    _wl_physx_component_get_sleepOnActivate: (id: number) => number;
    _wl_physx_component_set_massSpaceInertiaTensor: (id: number, x: number, y: number, z: number) => void;
    _wl_physx_component_addForce: (id: number, x: number, y: number, z: number, mode: number, localForce: boolean) => void;
    _wl_physx_component_addForceAt: (id: number, x: number, y: number, z: number, mode: number, localForce: boolean, posX: number, posY: number, posZ: number, local: boolean) => void;
    _wl_physx_component_addTorque: (id: number, x: number, y: number, z: number, mode: number) => void;
    _wl_physx_component_addCallback: (id: number, otherId: number) => number;
    _wl_physx_component_removeCallback: (id: number, callbackId: number) => number;
    _wl_physx_update_global_pose: (object: number, component: number) => void;
    _wl_physx_ray_cast: (scene: number, x: number, y: number, z: number, dx: number, dy: number, dz: number, groupMask: number, outPtr: number, maxDistance: number) => void;
    _wl_physx_set_collision_callback: (callback: number) => void;
    _wl_mesh_create: (indicesPtr: number, indicesSize: number, indexType: number, vertexCount: number, skinningType: number) => number;
    _wl_mesh_get_vertexCount: (index: number) => number;
    _wl_mesh_get_indexData: (index: number, outPtr: number, count: number) => number;
    _wl_mesh_update: (index: number) => void;
    _wl_mesh_get_boundingSphere: (index: number, outPtr: number) => void;
    _wl_mesh_get_attribute: (index: number, attribute: number, outPtr: number) => void;
    _wl_mesh_destroy: (index: number) => void;
    _wl_mesh_get_attribute_values: (attribute: number, srcFormatSize: number, srcPtr: number, srcStride: number, dstFormatSize: number, destPtr: number, dstSize: number) => void;
    _wl_mesh_set_attribute_values: (attribute: number, srcFormatSize: number, srcPtr: number, srcSize: number, dstFormatSize: number, destPtr: number, destStride: number) => void;
    _wl_font_get_emHeight: (index: number) => number;
    _wl_font_get_capHeight: (index: number) => number;
    _wl_font_get_xHeight: (index: number) => number;
    _wl_font_get_outlineSize: (index: number) => number;
    _wl_material_create: (definitionIndex: number) => number;
    _wl_material_get_definition: (index: number) => number;
    _wl_material_definition_get_param_count: (index: number) => number;
    _wl_material_definition_get_param_name: (index: number, paramIndex: number) => number;
    _wl_material_definition_get_param_type: (index: number, paramIndex: number) => number;
    _wl_material_get_pipeline: (index: number) => number;
    _wl_material_clone: (index: number) => number;
    _wl_material_get_param_index: (index: number, namePtr: number) => number;
    _wl_material_get_param_type: (index: number, paramIndex: number) => number;
    _wl_material_get_param_value: (index: number, paramIndex: number, outPtr: number) => number;
    _wl_material_set_param_value_uint: (index: number, paramId: number, valueId: number) => void;
    _wl_material_set_param_value_float: (index: number, paramId: number, ptr: number, count: number) => void;
    _wl_image_create: (jsImage: number, borderPadding: boolean) => number;
    _wl_image_destroy: (index: number) => number;
    _wl_image_size: (index: number, out: number) => number;
    _wl_image_get_jsImage_index: (index: number) => number;
    _wl_image_borderPadding: (index: number) => number;
    _wl_image_compressed: (index: number) => number;
    _wl_image_markDirty: (index: number) => void;
    _wl_image_markReady: (index: number, width: number, height: number, flipY: number) => void;
    _wl_texture_create: (image: number) => number;
    _wl_texture_get_image_index: (index: number) => number;
    _wl_texture_destroy: (id: number) => void;
    _wl_probeVolumeScenario_find: (filename: number) => number;
    _wl_probeVolumeScenario_loaded: (index: number) => number;
    _wl_resource_count: (type: number) => number;
    _wl_resource_destroyed: (type: number, index: number) => number;
    _wl_resource_originScene: (type: number, index: number) => number;
    _wl_renderer_updateImage: (imageIndex: number, jsImageIndex: number, srcWidth: number, srcHeight: number, xOffset: number, yOffset: number, flipY: boolean) => number;
    _wl_animation_get_duration: (id: number) => number;
    _wl_animation_get_trackCount: (id: number) => number;
    _wl_animation_retargetToSkin: (id: number, targetId: number) => number;
    _wl_animation_retarget: (id: number, ptr: number) => number;
    _wl_animationGraph_sampler_set_playbackSpeed: (id: number, index: number, speed: number) => void;
    _wl_animationGraph_create1DBlend: (animsPtr: number, animCount: number) => number;
    _wl_particleEffect_create: (maxCount: number, meshId: number, materialId: number, colors: boolean, instanceData: boolean) => number;
    _wl_particleEffect_get_maxCount: (index: number) => number;
    _wl_particleEffect_destroy: (index: number) => void;
    _wl_object_id: (scene: number, index: number) => number;
    _wl_object_index: (id: number) => number;
    _wl_object_name: (id: number) => number;
    _wl_object_set_name: (id: number, ptr: number) => void;
    _wl_object_remove: (id: number) => void;
    _wl_object_markedDestroyed: (id: number) => boolean;
    _wl_object_parent: (id: number) => number;
    _wl_object_get_children_count: (id: number) => number;
    _wl_object_get_children: (id: number, outPtr: number, count: number) => number;
    _wl_object_set_parent: (id: number, parentId: number) => void;
    _wl_object_clone: (id: number, parentId: number) => number;
    _wl_object_reset_scaling: (id: number) => void;
    _wl_object_reset_translation_rotation: (id: number) => void;
    _wl_object_reset_rotation: (id: number) => void;
    _wl_object_reset_translation: (id: number) => void;
    _wl_object_translate: (id: number, x: number, y: number, z: number) => void;
    _wl_object_translate_obj: (id: number, x: number, y: number, z: number) => void;
    _wl_object_translate_world: (id: number, x: number, y: number, z: number) => void;
    _wl_object_rotate_axis_angle: (id: number, x: number, y: number, z: number, deg: number) => void;
    _wl_object_rotate_axis_angle_rad: (id: number, x: number, y: number, z: number, rad: number) => void;
    _wl_object_rotate_axis_angle_obj: (id: number, x: number, y: number, z: number, deg: number) => void;
    _wl_object_rotate_axis_angle_rad_obj: (id: number, x: number, y: number, z: number, rad: number) => void;
    _wl_object_rotate_quat: (id: number, x: number, y: number, z: number, w: number) => void;
    _wl_object_rotate_quat_obj: (id: number, x: number, y: number, z: number, w: number) => void;
    _wl_object_scale: (id: number, x: number, y: number, z: number) => void;
    _wl_object_trans_local: (id: number) => number;
    _wl_object_get_translation_local: (id: number, outPtr: number) => void;
    _wl_object_set_translation_local: (id: number, x: number, y: number, z: number) => void;
    _wl_object_get_translation_world: (id: number, outPtr: number) => void;
    _wl_object_set_translation_world: (id: number, x: number, y: number, z: number) => void;
    _wl_object_trans_world: (id: number) => number;
    _wl_object_trans_world_to_local: (id: number) => number;
    _wl_object_scaling_local: (id: number) => number;
    _wl_object_scaling_world: (id: number) => number;
    _wl_object_set_scaling_local: (id: number, x: number, y: number, z: number) => void;
    _wl_object_set_scaling_world: (id: number, x: number, y: number, z: number) => void;
    _wl_object_scaling_world_to_local: (id: number) => number;
    _wl_object_set_rotation_local: (id: number, x: number, y: number, z: number, w: number) => void;
    _wl_object_set_rotation_world: (id: number, x: number, y: number, z: number, w: number) => void;
    _wl_object_transformVectorWorld: (id: number, ptr: number) => number;
    _wl_object_transformVectorLocal: (id: number, ptr: number) => number;
    _wl_object_transformPointWorld: (id: number, ptr: number) => number;
    _wl_object_transformPointLocal: (id: number, ptr: number) => number;
    _wl_object_transformVectorInverseWorld: (id: number, ptr: number) => number;
    _wl_object_transformVectorInverseLocal: (id: number, ptr: number) => number;
    _wl_object_transformPointInverseWorld: (id: number, ptr: number) => number;
    _wl_object_transformPointInverseLocal: (id: number, ptr: number) => number;
    _wl_object_toWorldSpaceTransform: (id: number, ptr: number) => number;
    _wl_object_toObjectSpaceTransform: (id: number, ptr: number) => number;
    _wl_object_lookAt: (id: number, x: number, y: number, z: number, upX: number, upY: number, upZ: number) => void;
    _wl_object_set_dirty: (id: number) => void;
    _wl_get_js_component_index: (id: number, outPtr: number, count: number) => number;
    _wl_get_js_component_index_for_id: (id: number) => number;
    _wl_get_component_id: (id: number, managerId: number, index: number) => number;
    _wl_object_get_components: (id: number, outPtr: number, count: number) => number;
    _wl_object_get_component_types: (id: number, outPtr: number, count: number) => void;
    _wl_object_add_js_component: (id: number, typeId: number) => number;
    _wl_object_add_component: (id: number, typeId: number) => number;
    _wl_object_is_changed: (id: number) => number;
    _wl_object_findByName: (obj: number, name: number, indexPtr: number, childPtr: number, outPtr: number, count: number) => number;
    _wl_object_findByNameRecursive: (obj: number, name: number, indexPtr: number, outPtr: number, count: number) => number;
    _wl_component_manager_name: (scene: number, id: number) => number;
    _wl_pipeline_get_name: (index: number) => number;
    _wl_pipeline_get_api_object: (index: number) => number;
    _wl_pipeline_findByName: (name: number) => number;
    _wl_pipeline_set_enableRendererHooks: (index: number, value: boolean) => void;
    _wl_pipeline_get_enableRendererHooks: (index: number) => boolean;
    _wl_skin_get_joint_count: (id: number) => number;
    _wl_skin_joint_ids: (id: number) => number;
    _wl_skin_inverse_bind_transforms: (id: number) => number;
    _wl_skin_inverse_bind_scalings: (id: number) => number;
    _wl_morph_targets_get_target_count: (id: number) => number;
    _wl_morph_targets_get_target_name: (id: number, target: number) => number;
    _wl_morph_targets_get_target_index: (id: number, namePtr: number) => number;
    _wl_morph_target_weights_get_weight: (id: number, target: number) => number;
    _wl_morph_target_weights_set_weight: (id: number, target: number, weight: number) => void;
    _wl_audioClip_source: (id: number) => number;
    _wl_math_cubicHermite: (a: number, b: number, c: number, d: number, f: number, e: number, isQuat: boolean) => void;
    _wl_i18n_setLanguage: (ptr: number) => void;
    _wl_i18n_currentLanguage: () => number;
    _wl_i18n_currentLanguageIndex: () => number;
    _wl_i18n_translate: (ptr: number) => number;
    _wl_i18n_languageCount: () => number;
    _wl_i18n_languageIndex: (ptr: number) => number;
    _wl_i18n_languageCode: (index: number) => number;
    _wl_i18n_languageName: (index: number) => number;
    _wl_i18n_languageFile: (index: number) => number;
}
