import { initialize } from '@zappar/zappar-cv';
import { mat4 } from 'gl-matrix';
import { VERSION } from './version.js';
let _z;
let _options;
export function setOptions(opts) {
    if (_z)
        console.warn('setOptions called after Zappar JS initialization');
    _options = Object.assign(Object.assign({}, _options), opts);
}
export function setPreferWebXRCamera(p) {
    if (_z)
        console.warn('setPreferWebXRCamera called after Zappar JS initialization');
    _options = Object.assign(Object.assign({}, _options), { preferWebXRCamera: p });
}
/**
 * @ignore
 */
export function z() {
    if (!_z) {
        console.log(`Zappar JS v${VERSION}`);
        _z = initialize(_options);
    }
    return _z;
}
/**
 * Gets the ID or the default rear- or user-facing camera.
 * @param userFacing - Whether 'selfie' camera ID should be returned.
 * @returns The camera device ID.
 * @see https://developer.mozilla.org/en-US/docs/Web/API/MediaDevices/enumerateDevices
 */
export function cameraDefaultDeviceID(userFacing) {
    return z().camera_default_device_id(userFacing || false);
}
/**
 * Inverts a 4x4 Float32Array Matrix.
 * @param m - The 4x4 matrix to be inverted.
 * @returns The inverted Float32Array matrix.
 */
export function invert(m) {
    const ret = mat4.create();
    mat4.invert(ret, m);
    return ret;
}
/**
 * Calculates the projection matrix from a given camera model (i.e. intrinsic camera parameters) and size of camera frame.
 *
 * @param model - The camera model.
 * @param cameraDataWidth - The width of the camera image (in pixels).
 * @param cameraDataHeight - The height of the camera image (in pixels).
 * @param renderWidth - The width of the canvas.
 * @param renderHeight - The height of the canvas.
 * @param zNear - The near clipping plane.
 * @param zFar - The far clipping plane.
 * @returns A 4x4 column-major projection matrix.
 */
export function projectionMatrixFromCameraModelAndSize(model, cameraDataWidth, cameraDataHeight, renderWidth, renderHeight, zNear = 0.1, zFar = 100) {
    return z().projection_matrix_from_camera_model_and_size_ext(model, cameraDataWidth, cameraDataHeight, renderWidth, renderHeight, zNear, zFar);
}
/**
 * @ignore
 */
export function drawPlane(gl, projectionMatrix, cameraMatrix, targetMatrix, texture) {
    z().draw_plane(gl, projectionMatrix, cameraMatrix, targetMatrix, texture);
}
/**
 * @ignore
 */
export function drawAxis(gl, projectionMatrix, cameraMatrix, targetMatrix) {
    z().draw_axis(gl, projectionMatrix, cameraMatrix, targetMatrix);
}
/**
 * Detects if your page is running in a browser that's not supported
 * @returns 'true' if the browser is incompatible.
 */
export function browserIncompatible() {
    return z().browser_incompatible();
}
/**
 * Shows a full-page dialog that informs the user they're using an unsupported browser,
 * and provides a button to 'copy' the current page URL so they can 'paste' it into the
 * address bar of a compatible alternative.
 */
export function browserIncompatibleUI() {
    z().browser_incompatible_ui();
}
/**
 * Check if the Zappar library is fully loaded and ready to process data.
 * Note that you can still use the full API before this function returns true - it's here to help implement loading screens.
 *
 * @returns 'true' if the library is fully loaded.
 */
export function loaded() {
    return z().loaded();
}
/**
 * Gets a promise that resolves when the Zappar library is fully loaded and ready to process data.
 * Note that you can still use the full API before this promise is resolved - it's here to help implement loading screens.
 *
 * @returns a promise that resolves when the library is fully loaded.
 */
export async function loadedPromise() {
    // eslint-disable-next-line no-constant-condition
    while (true) {
        if (loaded())
            return;
        await delay(50);
    }
}
function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}
/**
 * Pass true to this function if your user has consented to analytics cookies.
 * If set to true, a local storage item will store a pseudorandom identifier for the user for the purposes of
 * reporting 'unique user' analytics if the page is hosted using Zapworks.
 *
 * @param c - if cookies are permitted or not
 */
export function cookiesPermitted(c) {
    z().cookies_permitted(c);
}
/**
 * Convert a pose from the coordinate system of the raw camera data to that of the page,
 * taking into consideration screen rotation.
 *
 * @param rawPose - The raw pose in the coordinate system of the raw camera data
 * @param mirror - If the pose should be mirrored in the X axis
 */
export function poseFromRaw(rawPose, mirror) {
    return z().pose_from_raw(rawPose, mirror !== null && mirror !== void 0 ? mirror : false);
}
/**
 * Return true if the current page is running inside Zappar's App Clip / Instant App environment.
 */
export function isAppClip() {
    return z().in_app_clip();
}
