import { z } from './zappar.js';
/**
 * Plays back a previously recorded sequence of camera and motion data.
 */
export class SequenceSource {
    /**
     * Constructs a new SequenceSource.
     * @param _pipeline - The pipeline that this source will operate within.
     */
    constructor(pipeline) {
        this._z = z();
        this._impl = this._z.sequence_source_create(pipeline._getImpl());
    }
    /**
     * Destroys the sequence source.
     */
    destroy() {
        this._z.sequence_source_destroy(this._impl);
    }
    /**
     * Starts the sequence source.
     *
     * Starting a given source pauses any other sources within the same pipeline.
     */
    start() {
        this._z.sequence_source_start(this._impl);
    }
    /**
     * Pauses the sequence source.
     */
    pause() {
        this._z.sequence_source_pause(this._impl);
    }
    /**
     * Loads sequence data.
     * @param src - A URL to, or an ArrayBuffer of, the sequence data you'd like to play back.
     * @returns A promise that's resolved once the data is downloaded.
     */
    async load(src) {
        if (typeof src === 'string') {
            src = await (await fetch(src)).arrayBuffer();
        }
        this._z.sequence_source_load_from_memory(this._impl, src);
    }
    /**
     * Manually set the current time for the sequence.
     * @param t The time in ms, or `undefined` to use the system time.
     */
    setTime(t) {
        this._z.sequence_source_time_set(this._impl, t !== null && t !== void 0 ? t : -1);
    }
}
