import { instant_world_tracker_transform_orientation_t } from '@zappar/zappar-cv';
import { z } from './zappar.js';
/**
 * Attaches content to a point on a surface in front of the user as it moves around in the camera view.
 * @see https://docs.zap.works/universal-ar/javascript/instant-world-tracking/
 */
export class InstantWorldTracker {
    /**
     * Constructs a new InstantWorldTracker.
     * @param _pipeline - The pipeline that this tracker will operate within.
     */
    constructor(_pipeline) {
        this._pipeline = _pipeline;
        /**
         * The instant world tracking anchor.
         */
        this.anchor = {
            poseCameraRelative: mirror => this._anchorPoseCameraRelative(mirror),
            pose: (cameraPose, mirror) => this._anchorPose(cameraPose, mirror),
        };
        this._z = z();
        this._impl = this._z.instant_world_tracker_create(this._pipeline._getImpl());
    }
    /**
     * Destroys the instant tracker.
     */
    destroy() {
        this._z.instant_world_tracker_destroy(this._impl);
    }
    _anchorPoseCameraRelative(mirror) {
        return this._z.instant_world_tracker_anchor_pose_camera_relative(this._impl, mirror === true);
    }
    _anchorPose(cameraPose, mirror) {
        return this._z.instant_world_tracker_anchor_pose(this._impl, cameraPose, mirror === true);
    }
    /**
     * Gets/sets the enabled state of the instant world tracker.
     * Disable when not in use to save computational resources during frame processing.
     */
    get enabled() {
        return this._z.instant_world_tracker_enabled(this._impl);
    }
    set enabled(e) {
        this._z.instant_world_tracker_enabled_set(this._impl, e);
    }
    /**
     * Sets the point in the user's environment that the anchor tracks from.
     *
     * The parameters passed in to this function correspond to the X, Y and Z coordinates (in camera space) of the point to track. Choosing a position with X and Y coordinates of zero, and a negative Z coordinate, will select a point on a surface directly in front of the center of the screen.
     *
     * @param orientation -  The orientation of the point in space.
     */
    setAnchorPoseFromCameraOffset(x, y, z, orientation) {
        this._z.instant_world_tracker_anchor_pose_set_from_camera_offset(this._impl, x, y, z, orientation || instant_world_tracker_transform_orientation_t.MINUS_Z_AWAY_FROM_USER);
    }
}
