import { Event, Event1 } from './event.js';
import { z } from './zappar.js';
/**
 * Attaches content to a face as it moves around in the camera view.
 * @see https://docs.zap.works/universal-ar/javascript/face-tracking/
 */
export class FaceTracker {
    /**
     * Constructs a new FaceTracker
     * @param _pipeline - The pipeline that this tracker will operate within.
     */
    constructor(_pipeline) {
        this._pipeline = _pipeline;
        /**
         * Emitted when an anchor becomes visible in a camera frame.
         */
        this.onVisible = new Event1();
        /**
         * Emitted when an anchor goes from being visible in the previous camera frame, to not being visible in the current frame.
         */
        this.onNotVisible = new Event1();
        /**
         * Emitted when a new anchor is created by the tracker.
         */
        this.onNewAnchor = new Event1();
        /**
         * The set of currently visible anchors.
         */
        this.visible = new Set();
        /**
         * A map of the available anchors by their respective IDs.
         */
        this.anchors = new Map();
        this._visibleLastFrame = new Set();
        this._frameUpdate = () => {
            const newAnchors = new Set();
            // Swap the visible and visibleLastFrame so we can avoid a set allocation
            const swap = this.visible;
            this.visible = this._visibleLastFrame;
            this._visibleLastFrame = swap;
            this.visible.clear();
            const num = this._z.face_tracker_anchor_count(this._impl);
            for (let i = 0; i < num; i++) {
                const id = this._z.face_tracker_anchor_id(this._impl, i);
                let anchor = this.anchors.get(id);
                let isNew = false;
                if (!anchor) {
                    anchor = {
                        onVisible: new Event(),
                        onNotVisible: new Event(),
                        indx: 0,
                        id: id,
                        poseCameraRelative: mirror => this._z.face_tracker_anchor_pose_camera_relative(this._impl, anchor.indx, mirror === true),
                        pose: (cameraPose, mirror) => this._z.face_tracker_anchor_pose(this._impl, anchor.indx, cameraPose, mirror === true),
                        identity: new Float32Array(50),
                        expression: new Float32Array(29),
                        visible: true,
                    };
                    isNew = true;
                    this.anchors.set(id, anchor);
                    newAnchors.add(anchor);
                }
                anchor.indx = i;
                anchor.visible = true;
                anchor.identity = this._z.face_tracker_anchor_identity_coefficients(this._impl, i);
                anchor.expression = this._z.face_tracker_anchor_expression_coefficients(this._impl, i);
                this.visible.add(anchor);
            }
            // Events
            for (const anchor of newAnchors)
                this.onNewAnchor.emit(anchor);
            for (const anchor of this.visible) {
                if (!this._visibleLastFrame.has(anchor)) {
                    this.onVisible.emit(anchor);
                    anchor.onVisible.emit();
                }
                else {
                    this._visibleLastFrame.delete(anchor);
                }
            }
            for (const anchor of this._visibleLastFrame) {
                anchor.visible = false;
                this.onNotVisible.emit(anchor);
                anchor.onNotVisible.emit();
            }
        };
        this._pipeline._onFrameUpdateInternal.bind(this._frameUpdate);
        this._z = z();
        this._impl = this._z.face_tracker_create(this._pipeline._getImpl());
    }
    /**
     * Destroys the face tracker.
     */
    destroy() {
        this._pipeline._onFrameUpdateInternal.unbind(this._frameUpdate);
        this.anchors.clear();
        this.visible.clear();
        this._z.face_tracker_destroy(this._impl);
    }
    /**
     * Loads face tracking model data.
     * @param src - A URL to, or ArrayBuffer of, model data.
     * @returns A promise that's resolved once the model is loaded. It may still take a few frames for the tracker to fully initialize and detect faces.
     */
    async loadModel(src) {
        if (typeof src === 'string') {
            src = await (await fetch(src)).arrayBuffer();
        }
        this._z.face_tracker_model_load_from_memory(this._impl, src);
    }
    /**
     * Loads the default face tracking model.
     * @returns A promise that's resolved once the model is loaded. It may still take a few frames for the tracker to fully initialize and detect faces.
     */
    async loadDefaultModel() {
        await this._z.face_tracker_model_load_default(this._impl);
    }
    /**
     * Gets/sets the enabled state of the face tracker.
     * Disable when not in use to save computational resources during frame processing.
     */
    get enabled() {
        return this._z.face_tracker_enabled(this._impl);
    }
    set enabled(e) {
        this._z.face_tracker_enabled_set(this._impl, e);
    }
    /**
     * Gets/sets the maximum number of faces to track.
     *
     * By default only one face is tracked in any given frame. Increasing this number may reduce runtime performance.
     */
    get maxFaces() {
        return this._z.face_tracker_max_faces(this._impl);
    }
    set maxFaces(m) {
        this._z.face_tracker_max_faces_set(this._impl, m);
    }
}
