import { z } from './zappar.js';
/**
 * A mesh that fits to the user's face and deforms as the user's expression changes.
 * @see https://docs.zap.works/universal-ar/javascript/face-tracking/
 */
export class FaceMesh {
    /**
     * Constructs a new FaceMesh.
     * @see https://docs.zap.works/universal-ar/javascript/face-tracking/
     */
    constructor() {
        this._z = z();
        this._impl = this._z.face_mesh_create();
    }
    /**
     * Destroys the face mesh.
     */
    destroy() {
        this._z.face_mesh_destroy(this._impl);
    }
    /**
     * Loads the data for a face mesh.
     * @param src - A URL or ArrayBuffer of the source mesh data.
     * @param fillMouth - If true, fills this face feature with polygons.
     * @param fillEyeLeft - If true, fills this face feature with polygons.
     * @param fillEyeRight - If true, fills this face feature with polygons.
     * @param fillNeck - If true, fills this face feature with polygons.
     * @returns A promise that's resolved once the data is loaded.
     */
    async load(src, fillMouth, fillEyeLeft, fillEyeRight, fillNeck) {
        if (!src) {
            this.loadDefault();
            return;
        }
        else if (typeof src === 'string') {
            src = await (await fetch(src)).arrayBuffer();
        }
        this._z.face_mesh_load_from_memory(this._impl, src, fillMouth || false, fillEyeLeft || false, fillEyeRight || false, fillNeck || false);
    }
    /**
     * Loads the default face mesh data.
     * @returns A promise that's resolved once the data is loaded.
     */
    async loadDefault() {
        await this._z.face_mesh_load_default(this._impl);
    }
    /**
     * Loads the default face mesh.
     * @param fillMouth - If true, fills this face feature with polygons.
     * @param fillEyeLeft - If true, fills this face feature with polygons.
     * @param fillEyeRight - If true, fills this face feature with polygons.
     * @returns A promise that's resolved once the data is loaded.
     */
    async loadDefaultFace(fillMouth, fillEyeLeft, fillEyeRight) {
        await this._z.face_mesh_load_default_face(this._impl, fillMouth || false, fillEyeLeft || false, fillEyeRight || false);
    }
    /**
     * The full head simplified mesh covers the whole of the user's head, including some neck.
     * It's ideal for drawing into the depth buffer in order to mask out the back of 3D models placed on the user's head.
     * @param fillMouth - If true, fills this face feature with polygons.
     * @param fillEyeLeft - If true, fills this face feature with polygons.
     * @param fillEyeRight - If true, fills this face feature with polygons.
     * @param fillNeck - If true, fills this face feature with polygons.
     * @returns A promise that's resolved once the data is loaded.
     */
    async loadDefaultFullHeadSimplified(fillMouth, fillEyeLeft, fillEyeRight, fillNeck) {
        await this._z.face_mesh_load_default_full_head_simplified(this._impl, fillMouth || false, fillEyeLeft || false, fillEyeRight || false, fillNeck || false);
    }
    /**
     * Update the face mesh directly from a [[FaceAnchor]].
     * @param f - The face anchor.
     * @param mirror - Pass `true` to mirror the location in the X-axis.
     */
    updateFromFaceAnchor(f, mirror) {
        this._z.face_mesh_update(this._impl, f.identity, f.expression, mirror || false);
    }
    /**
     * Updates the face mesh directly from a identity and expression coefficients.
     * @param identity - The identity coefficients.
     * @param expression - The expression coefficients.
     * @param mirror - Pass `true` to mirror the location in the X-axis.
     */
    updateFromIdentityExpression(identity, expression, mirror) {
        this._z.face_mesh_update(this._impl, identity, expression, mirror || false);
    }
    /**
     *
     * @returns The vertices of the mesh.
     */
    get vertices() {
        return this._z.face_mesh_vertices(this._impl);
    }
    /**
     * @returns The indices of the mesh.
     */
    get indices() {
        return this._z.face_mesh_indices(this._impl);
    }
    /**
     * @returns The UVs of the mesh.
     */
    get uvs() {
        return this._z.face_mesh_uvs(this._impl);
    }
    /**
     * @returns The normals of the mesh.
     */
    get normals() {
        return this._z.face_mesh_normals(this._impl);
    }
    /**
     * @ignore
     */
    _getImpl() {
        return this._impl;
    }
}
