import { Event, Event1 } from './event.js';
import { z } from './zappar.js';
export var D3Type;
(function (D3Type) {
    D3Type[D3Type["QR"] = 0] = "QR";
    D3Type[D3Type["DENSE"] = 1] = "DENSE";
})(D3Type || (D3Type = {}));
/**
 * Attaches content to a D3 Marker
 */
export class D3Tracker {
    constructor(_pipeline) {
        this._pipeline = _pipeline;
        /**
         * Emitted when an anchor becomes visible in a camera frame.
         */
        this.onVisible = new Event1();
        /**
         * Emitted when an anchor goes from being visible in the previous camera frame, to not being visible in the current frame.
         */
        this.onNotVisible = new Event1();
        /**
         * Emitted when a new anchor is created by the tracker.
         */
        this.onNewAnchor = new Event1();
        /**
         * The set of currently visible anchors.
         */
        this.visible = new Set();
        this.anchors = new Map();
        this._visibleLastFrame = new Set();
        this._processMaxResolution = false;
        this._frameUpdate = () => {
            const newAnchors = new Set();
            // Swap the visible and visibleLastFrame so we can avoid a set allocation
            const swap = this.visible;
            this.visible = this._visibleLastFrame;
            this._visibleLastFrame = swap;
            this.visible.clear();
            const num = this._z.d3_tracker_count(this._impl);
            for (let i = 0; i < num; i++) {
                const id = this._z.d3_tracker_id(this._impl, i);
                let anchor = this.anchors.get(id);
                let isNew = false; // TODO: declared but never used?
                if (!anchor) {
                    if (this._z.d3_tracker_type(this._impl, i) === D3Type.QR) {
                        anchor = {
                            id,
                            type: D3Type.QR,
                            poseCameraRelative: mirror => this._z.d3_tracker_pose_camera_relative(this._impl, i, mirror === true),
                            pose: (cameraPose, mirror) => this._z.d3_tracker_pose(this._impl, i, cameraPose, mirror === true),
                            visible: true,
                            url: this._z.d3_tracker_qr(this._impl, i),
                            category: this._z.d3_tracker_category(this._impl, i),
                            onVisible: new Event(),
                            onNotVisible: new Event(),
                            pose0CameraRelative: mirror => this._z.d3_tracker_pose0_camera_relative(this._impl, i, mirror === true),
                            pose0: (cameraPose, mirror) => this._z.d3_tracker_pose0(this._impl, i, cameraPose, mirror === true),
                            pose1CameraRelative: mirror => this._z.d3_tracker_pose1_camera_relative(this._impl, i, mirror === true),
                            pose1: (cameraPose, mirror) => this._z.d3_tracker_pose1(this._impl, i, cameraPose, mirror === true),
                        };
                    }
                    else {
                        anchor = {
                            id,
                            type: D3Type.DENSE,
                            poseCameraRelative: mirror => this._z.d3_tracker_pose_camera_relative(this._impl, i, mirror === true),
                            pose: (cameraPose, mirror) => this._z.d3_tracker_pose(this._impl, i, cameraPose, mirror === true),
                            pose0CameraRelative: mirror => this._z.d3_tracker_pose0_camera_relative(this._impl, i, mirror === true),
                            pose0: (cameraPose, mirror) => this._z.d3_tracker_pose0(this._impl, i, cameraPose, mirror === true),
                            pose1CameraRelative: mirror => this._z.d3_tracker_pose1_camera_relative(this._impl, i, mirror === true),
                            pose1: (cameraPose, mirror) => this._z.d3_tracker_pose1(this._impl, i, cameraPose, mirror === true),
                            visible: true,
                            value: this._z.d3_tracker_dense(this._impl, i),
                            onVisible: new Event(),
                            onNotVisible: new Event(),
                        };
                    }
                    isNew = true;
                    this.anchors.set(id, anchor);
                    newAnchors.add(anchor);
                }
                if (anchor.type === D3Type.QR) {
                    anchor.url = this._z.d3_tracker_qr(this._impl, i);
                    anchor.category = this._z.d3_tracker_category(this._impl, i);
                }
                anchor.poseCameraRelative = mirror => this._z.d3_tracker_pose_camera_relative(this._impl, i, mirror === true);
                anchor.pose = (cameraPose, mirror) => this._z.d3_tracker_pose(this._impl, i, cameraPose, mirror === true);
                anchor.pose0CameraRelative = mirror => this._z.d3_tracker_pose0_camera_relative(this._impl, i, mirror === true);
                anchor.pose0 = (cameraPose, mirror) => this._z.d3_tracker_pose0(this._impl, i, cameraPose, mirror === true);
                anchor.pose1CameraRelative = mirror => this._z.d3_tracker_pose1_camera_relative(this._impl, i, mirror === true);
                anchor.pose1 = (cameraPose, mirror) => this._z.d3_tracker_pose1(this._impl, i, cameraPose, mirror === true);
                anchor.visible = true;
                this.visible.add(anchor);
            }
            // Events
            for (const anchor of newAnchors)
                this.onNewAnchor.emit(anchor);
            for (const anchor of this.visible) {
                if (!this._visibleLastFrame.has(anchor)) {
                    this.onVisible.emit(anchor);
                    anchor.onVisible.emit();
                }
                else {
                    this._visibleLastFrame.delete(anchor);
                }
            }
            for (const anchor of this._visibleLastFrame) {
                anchor.visible = false;
                this.onNotVisible.emit(anchor);
                anchor.onNotVisible.emit();
            }
        };
        this._pipeline._onFrameUpdateInternal.bind(this._frameUpdate);
        this._z = z();
        this._impl = this._z.d3_tracker_create(this._pipeline._getImpl());
    }
    destroy() {
        this._pipeline._onFrameUpdateInternal.unbind(this._frameUpdate);
        this.anchors.clear();
        this.visible.clear();
        this._z.d3_tracker_destroy(this._impl);
    }
    /**
     * Gets/sets the enabled state of the image tracker.
     * Disable when not in use to save computational resources during frame processing.
     */
    get enabled() {
        return this._z.d3_tracker_enabled(this._impl);
    }
    set enabled(e) {
        this._z.d3_tracker_enabled_set(this._impl, e);
    }
    set processMaxResolution(value) {
        this._processMaxResolution = value;
        this._z.d3_tracker_process_max_resolution_set(this._impl, value);
    }
    get processMaxResolution() {
        return this._processMaxResolution;
    }
    setSizeForID(id, size) {
        this._z.d3_tracker_size_for_id_set(this._impl, id, size);
    }
}
