import { camera_profile_t } from "./gen/zappar";
import { Pipeline } from "./pipeline";
import { Source } from "./source";
import { profile } from "./profile";
import { mat4 } from "gl-matrix";
import { zcout, zcerr } from "./loglevel";
import { ImageProcessGL } from "./image-process-gl";
let latest = 1;
let byId = new Map();
export class HTMLElementSource extends Source {
    constructor(_video, _pipeline) {
        super();
        this._video = _video;
        this._pipeline = _pipeline;
        this._isPaused = true;
        this._hadFrames = false;
        this._isUserFacing = false;
        this._cameraToScreenRotation = 0;
        this._isUploadFrame = true;
        this._cameraToDeviceTransform = mat4.create();
        this._cameraToDeviceTransformUserFacing = mat4.create();
        this._cameraModel = new Float32Array([300, 300, 160, 120, 0, 0]);
        this._profile = camera_profile_t.DEFAULT;
        this._waitingForFrame = false;
        this._rgbOffscreenContext = null;
        this._rgbDomContext = null;
        this._isVideoElement = false;
        this._lastPresentedFrames = -1;
        mat4.fromScaling(this._cameraToDeviceTransformUserFacing, [-1, 1, -1]);
        let video = this._video;
        this._isVideoElement = video instanceof HTMLVideoElement;
        if (this._isVideoElement) {
            video.addEventListener("loadedmetadata", () => { this._hadFrames = true; });
        }
        else {
            this._hadFrames = true;
        }
        this._resetGLContext = this._resetGLContext.bind(this);
        let p = Pipeline.get(this._pipeline);
        if (p)
            p.onGLContextReset.bind(this._resetGLContext);
    }
    static createVideoElementSource(p, element) {
        let ret = (latest++);
        byId.set(ret, new HTMLElementSource(element, p));
        zcout("html_element_source_t initialized");
        return ret;
    }
    static getVideoElementSource(m) {
        return byId.get(m);
    }
    _resetGLContext() {
        var _a, _b;
        this._currentVideoTexture = undefined;
        (_b = (_a = this._imageProcessor) === null || _a === void 0 ? void 0 : _a.resetGLContext) === null || _b === void 0 ? void 0 : _b.call(_a);
    }
    destroy() {
        let p = Pipeline.get(this._pipeline);
        if (p)
            p.onGLContextReset.unbind(this._resetGLContext);
        this.pause();
        this._resetGLContext();
        this._currentRgbFrame = undefined;
        this._rgbDomCanvas = undefined;
        this._rgbDomContext = null;
        this._rgbOffscreenCanvas = undefined;
        this._rgbOffscreenContext = null;
    }
    pause() {
        this._isPaused = true;
        let p = Pipeline.get(this._pipeline);
        if (p && p.currentCameraSource === this)
            p.currentCameraSource = undefined;
    }
    start() {
        var _a;
        if (this._isPaused) {
            this._isUploadFrame = true;
            if (this._video instanceof HTMLVideoElement)
                this._hadFrames = false;
        }
        this._isPaused = false;
        let p = Pipeline.get(this._pipeline);
        if (p && p.currentCameraSource !== this) {
            (_a = p.currentCameraSource) === null || _a === void 0 ? void 0 : _a.pause();
            p.currentCameraSource = this;
        }
    }
    getFrame(currentlyProcessing) {
        let pipeline = Pipeline.get(this._pipeline);
        if (!pipeline)
            return;
        let gl = pipeline.glContext;
        if (!gl)
            return;
        if (this._isPaused)
            return;
        if (!this._hadFrames)
            return;
        try {
            let info = this._processFrame(gl, this._cameraToScreenRotation, currentlyProcessing);
            if (info) {
                let token = pipeline.registerToken(info);
                // Use -1 for frame time for now since iOS accuracy is not good enough
                pipeline.sendDataToWorker(info.data || new ArrayBuffer(0), token, info.dataWidth, info.dataHeight, info.userFacing, info.cameraToDevice, info.cameraModel, -1); // this._frameTime ?? (performance.now() * 1000));
                info.data = undefined;
            }
        }
        catch (ex) {
            console.log("Unable to process frame");
        }
        return;
    }
    _requestVideoFrameCallback() {
        const video = this._video;
        if (!(video === null || video === void 0 ? void 0 : video.requestVideoFrameCallback))
            return;
        video.requestVideoFrameCallback((time, metadata) => {
            if (typeof metadata === 'object')
                this._frameTime = metadata.captureTime ? metadata.captureTime * 1000.0 : undefined;
            if (typeof metadata === 'object' && typeof metadata.presentedFrames === 'number') {
                if (this._lastPresentedFrames === metadata.presentedFrames) {
                    console.log('Avoided identical frame');
                    this._requestVideoFrameCallback();
                    return;
                }
                else {
                    this._lastPresentedFrames = metadata.presentedFrames;
                }
            }
            this._waitingForFrame = false;
        });
        this._waitingForFrame = true;
    }
    _processFrame(gl, rotation, currentlyProcessing) {
        let pipeline = Pipeline.get(this._pipeline);
        if (!pipeline)
            return undefined;
        if (!this._imageProcessor)
            this._imageProcessor = new ImageProcessGL(gl);
        if (this._isUploadFrame) {
            if (this._waitingForFrame)
                return;
            if (!this._currentVideoTexture) {
                this._currentVideoTexture = pipeline.getVideoTexture();
            }
            if (!this._currentVideoTexture)
                return undefined;
            if (pipeline.getCameraFrameDataRGBEnabled())
                this._currentRgbFrame = this._generateRGBFrame();
            this._imageProcessor.uploadFrame(this._currentVideoTexture, this._video, rotation, this._isUserFacing, this._profile);
            this._requestVideoFrameCallback();
            this._isUploadFrame = !this._isUploadFrame;
            return undefined;
        }
        if (currentlyProcessing || !this._currentVideoTexture)
            return undefined;
        this._isUploadFrame = !this._isUploadFrame;
        const [dataWidth, dataHeight] = profile.getDataSize(this._profile);
        let greySize = dataWidth * dataHeight;
        let pixels = pipeline.cameraPixelArrays.pop();
        while (pixels) {
            if (pixels.byteLength === greySize)
                break;
            pixels = pipeline.cameraPixelArrays.pop();
        }
        if (!pixels) {
            pixels = new ArrayBuffer(greySize);
        }
        let tex = this._currentVideoTexture;
        this._currentVideoTexture = undefined;
        let focalLength = (this._isUserFacing ? 300.0 : 240.0) * dataWidth / 320.0;
        this._cameraModel[0] = focalLength;
        this._cameraModel[1] = focalLength;
        this._cameraModel[2] = dataWidth * 0.5;
        this._cameraModel[3] = dataHeight * 0.5;
        const info = Object.assign(Object.assign({}, this._imageProcessor.readFrame(tex, pixels, this._profile)), { cameraModel: this._cameraModel, cameraSource: this, cameraToDevice: (this._isUserFacing ? this._cameraToDeviceTransformUserFacing : this._cameraToDeviceTransform) });
        if (this._currentRgbFrame) {
            info.rgbFrame = this._currentRgbFrame;
            this._currentRgbFrame = undefined;
        }
        return info;
    }
    uploadGL() {
        // No-op as already uploaded
    }
    setProfile(p) {
        this._profile = p;
    }
    _generateRGBFrame() {
        const width = this._isVideoElement ? this._video.videoWidth : this._video.naturalWidth;
        const height = this._isVideoElement ? this._video.videoHeight : this._video.naturalHeight;
        if (!(width > 0 && height > 0))
            return undefined;
        return profile.offscreenCanvasSupported
            ? this._generateOffscreenRgbFrame(width, height)
            : this._generateDomRgbFrame(width, height);
    }
    _generateOffscreenRgbFrame(width, height) {
        var _a;
        if (!this._rgbOffscreenCanvas) {
            this._rgbOffscreenCanvas = new OffscreenCanvas(width, height);
            this._rgbOffscreenContext = this._rgbOffscreenCanvas.getContext('2d', {
                alpha: false,
                willReadFrequently: true
            });
        }
        else if (this._rgbOffscreenCanvas.width !== width || this._rgbOffscreenCanvas.height !== height) {
            this._rgbOffscreenCanvas.width = width;
            this._rgbOffscreenCanvas.height = height;
        }
        (_a = this._rgbOffscreenContext) === null || _a === void 0 ? void 0 : _a.drawImage(this._video, 0, 0, width, height);
        try {
            return this._rgbOffscreenCanvas.transferToImageBitmap();
        }
        catch (e) {
            zcerr(`Failed to generate RGB frame: ${e}`);
            return undefined;
        }
    }
    _generateDomRgbFrame(width, height) {
        var _a, _b;
        if (!this._rgbDomCanvas) {
            this._rgbDomCanvas = document.createElement('canvas');
            this._rgbDomContext = this._rgbDomCanvas.getContext('2d', {
                alpha: false,
                willReadFrequently: true
            });
        }
        else if (this._rgbDomCanvas.width !== width || this._rgbDomCanvas.height !== height) {
            this._rgbDomCanvas.width = width;
            this._rgbDomCanvas.height = height;
        }
        (_a = this._rgbDomContext) === null || _a === void 0 ? void 0 : _a.drawImage(this._video, 0, 0, width, height);
        try {
            return (_b = this._rgbDomContext) === null || _b === void 0 ? void 0 : _b.getImageData(0, 0, width, height);
        }
        catch (e) {
            zcerr(`Failed to generate RGB frame: ${e}`);
            return undefined;
        }
    }
}
