import { compileShader, linkProgram } from "./shader";
export class MeshDraw {
    constructor(_gl) {
        this._gl = _gl;
    }
    dispose() {
        if (this._vbo)
            this._gl.deleteBuffer(this._vbo);
        if (this._uvbo)
            this._gl.deleteBuffer(this._uvbo);
        if (this._ibo)
            this._gl.deleteBuffer(this._ibo);
        if (this._shader)
            this._gl.deleteProgram(this._shader.prog);
        this._vbo = undefined;
        this._uvbo = undefined;
        this._ibo = undefined;
        this._shader = undefined;
    }
    _generateIBO(indices, gl) {
        if (this._ibo && this._lastIndices === indices)
            return this._ibo;
        this._lastIndices = indices;
        if (!this._ibo)
            this._ibo = gl.createBuffer();
        if (!this._ibo)
            throw new Error("Unable to create buffer object");
        gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, this._ibo);
        gl.bufferData(gl.ELEMENT_ARRAY_BUFFER, indices, gl.DYNAMIC_DRAW);
        gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
        return this._ibo;
    }
    _generateVBO(face, gl) {
        if (!this._vbo)
            this._vbo = gl.createBuffer();
        if (!this._vbo)
            throw new Error("Unable to create buffer object");
        gl.bindBuffer(gl.ARRAY_BUFFER, this._vbo);
        gl.bufferData(gl.ARRAY_BUFFER, face, gl.DYNAMIC_DRAW);
        gl.bindBuffer(gl.ARRAY_BUFFER, null);
        return this._vbo;
    }
    draw(projectionMatrix, cameraMatrix, targetMatrix, vertices, indices, vertexOffset, vertexStride, indicesCount) {
        let gl = this._gl;
        let shader = this._getShader(gl);
        let v = this._generateVBO(vertices, gl);
        let i = this._generateIBO(indices, gl);
        gl.disable(gl.DEPTH_TEST);
        gl.disable(gl.CULL_FACE);
        gl.useProgram(shader.prog);
        gl.uniformMatrix4fv(shader.unif_proj, false, projectionMatrix);
        gl.uniformMatrix4fv(shader.unif_camera, false, cameraMatrix);
        gl.uniformMatrix4fv(shader.unif_matrix, false, targetMatrix);
        gl.bindBuffer(gl.ARRAY_BUFFER, v);
        gl.vertexAttribPointer(shader.attr_position, 3, gl.FLOAT, false, vertexStride, vertexOffset);
        gl.enableVertexAttribArray(shader.attr_position);
        gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, i);
        gl.drawElements(gl.LINE_STRIP, indicesCount, gl.UNSIGNED_INT, 0);
        gl.disableVertexAttribArray(shader.attr_position);
        gl.bindBuffer(gl.ARRAY_BUFFER, null);
        gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, null);
    }
    _getShader(gl) {
        if (this._shader)
            return this._shader;
        let prog = gl.createProgram();
        if (!prog)
            throw new Error("Unable to create program");
        let vertexShader = compileShader(gl, gl.VERTEX_SHADER, vertexShaderSrc);
        let fragmentShader = compileShader(gl, gl.FRAGMENT_SHADER, fragmentShaderSrc);
        gl.attachShader(prog, vertexShader);
        gl.attachShader(prog, fragmentShader);
        linkProgram(gl, prog);
        let unif_proj = gl.getUniformLocation(prog, "projMatrix");
        if (!unif_proj)
            throw new Error("Unable to get uniform location projMatrix");
        let unif_matrix = gl.getUniformLocation(prog, "modelViewMatrix");
        if (!unif_matrix)
            throw new Error("Unable to get uniform location modelViewMatrix");
        let unif_camera = gl.getUniformLocation(prog, "cameraMatrix");
        if (!unif_camera)
            throw new Error("Unable to get uniform location cameraMatrix");
        this._shader = {
            prog,
            unif_matrix,
            unif_camera,
            unif_proj,
            attr_position: gl.getAttribLocation(prog, "position"),
        };
        return this._shader;
    }
}
let vertexShaderSrc = `
#ifndef GL_ES
#define highp
#define mediump
#define lowp
#endif

uniform mat4 projMatrix;
uniform mat4 cameraMatrix;
uniform mat4 modelViewMatrix;
attribute vec4 position;

void main()
{
    gl_Position = projMatrix * cameraMatrix * modelViewMatrix * position;
}`;
let fragmentShaderSrc = `
#define highp mediump
#ifdef GL_ES
    // define default precision for float, vec, mat.
    precision highp float;
#else
#define highp
#define mediump
#define lowp
#endif

void main()
{
    gl_FragColor = vec4(1.0, 0.0, 0.0, 0.5);
}`;
