import { compileShader, linkProgram } from "./shader";
let shader;
let vbo;
let uvbo;
const gridSize = 10;
export function disposeDrawGrid() {
    shader = undefined;
    vbo = undefined;
    uvbo = undefined;
}
function generate(gl) {
    if (vbo)
        return vbo;
    vbo = gl.createBuffer();
    if (!vbo)
        throw new Error("Unable to create buffer object");
    let coords = [
        -0.5, 0, 0.5,
        -0.5, 0, -0.5,
        0.5, 0, 0.5,
        0.5, 0, -0.5
    ];
    coords = coords.map(entry => entry * gridSize);
    gl.bindBuffer(gl.ARRAY_BUFFER, vbo);
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(coords), gl.STATIC_DRAW);
    gl.bindBuffer(gl.ARRAY_BUFFER, null);
    return vbo;
}
function generateUVBO(gl) {
    if (uvbo)
        return uvbo;
    uvbo = gl.createBuffer();
    if (!uvbo)
        throw new Error("Unable to create buffer object");
    let coords = [
        0, 1,
        0, 0,
        1, 1,
        1, 0
    ];
    coords = coords.map(entry => entry * gridSize);
    gl.bindBuffer(gl.ARRAY_BUFFER, uvbo);
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(coords), gl.STATIC_DRAW);
    gl.bindBuffer(gl.ARRAY_BUFFER, null);
    return uvbo;
}
export function drawGrid(gl, projectionMatrix, cameraMatrix, targetMatrix) {
    let shader = getShader(gl);
    let v = generate(gl);
    let uvbo = generateUVBO(gl);
    gl.disable(gl.DEPTH_TEST);
    gl.useProgram(shader.prog);
    gl.uniformMatrix4fv(shader.unif_proj, false, projectionMatrix);
    gl.uniformMatrix4fv(shader.unif_camera, false, cameraMatrix);
    gl.uniformMatrix4fv(shader.unif_matrix, false, targetMatrix);
    gl.bindBuffer(gl.ARRAY_BUFFER, v);
    gl.vertexAttribPointer(shader.attr_position, 3, gl.FLOAT, false, 3 * 4, 0);
    gl.enableVertexAttribArray(shader.attr_position);
    gl.bindBuffer(gl.ARRAY_BUFFER, uvbo);
    gl.vertexAttribPointer(shader.attr_textureCoord, 2, gl.FLOAT, false, 2 * 4, 0);
    gl.enableVertexAttribArray(shader.attr_textureCoord);
    gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);
    gl.disableVertexAttribArray(shader.attr_position);
    gl.disableVertexAttribArray(shader.attr_textureCoord);
    gl.bindBuffer(gl.ARRAY_BUFFER, null);
}
let vertexShaderSrc = `
#ifndef GL_ES
#define highp
#define mediump
#define lowp
#endif

uniform mat4 projMatrix;
uniform mat4 cameraMatrix;
uniform mat4 modelViewMatrix;
attribute vec4 position;
attribute vec2 textureCoord;

varying highp vec2 vTextureCoord;

void main()
{
    gl_Position = projMatrix * cameraMatrix * modelViewMatrix * position;
    vTextureCoord = textureCoord;
}`;
let fragmentShaderSrc = `
#define highp mediump
#ifdef GL_ES
    // define default precision for float, vec, mat.
    precision highp float;
#else
#define highp
#define mediump
#define lowp
#endif

varying highp vec2 vTextureCoord;
uniform sampler2D skinSampler;

void main()
{
    vec2 pixels = 500.0 * vTextureCoord + 0.5;
    if ((int(mod(pixels.x, 50.0)) == 0) || (int(mod(pixels.y, 50.0)) == 0)) {
        gl_FragColor = vec4(0.0, 0.0, 0.0, 1.0);
    } else {
        discard;
    }
}`;
function getShader(gl) {
    if (shader)
        return shader;
    let prog = gl.createProgram();
    if (!prog)
        throw new Error("Unable to create program");
    let vertexShader = compileShader(gl, gl.VERTEX_SHADER, vertexShaderSrc);
    let fragmentShader = compileShader(gl, gl.FRAGMENT_SHADER, fragmentShaderSrc);
    gl.attachShader(prog, vertexShader);
    gl.attachShader(prog, fragmentShader);
    linkProgram(gl, prog);
    let unif_proj = gl.getUniformLocation(prog, "projMatrix");
    if (!unif_proj)
        throw new Error("Unable to get uniform location projMatrix");
    let unif_matrix = gl.getUniformLocation(prog, "modelViewMatrix");
    if (!unif_matrix)
        throw new Error("Unable to get uniform location modelViewMatrix");
    let unif_camera = gl.getUniformLocation(prog, "cameraMatrix");
    if (!unif_camera)
        throw new Error("Unable to get uniform location cameraMatrix");
    shader = {
        prog,
        unif_matrix,
        unif_proj,
        unif_camera,
        attr_position: gl.getAttribLocation(prog, "position"),
        attr_textureCoord: gl.getAttribLocation(prog, "textureCoord")
    };
    return shader;
}
