import { Component, Object3D, WonderlandEngine } from '@wonderlandengine/api';
import { vec3 } from 'gl-matrix';
import { PlayerControllerInput } from './player-controller-input.js';
/**
 * The type of rotation to use when turning the player
 */
export declare enum RotationType {
    None = 0,
    /**
     * Snap will rotate by a fix amount of degrees in a row.
     *
     * It's the most comfortable for most people.
     */
    Snap = 1,
    /**
     * Smooth rotation over multiple frame.
     *
     * Smooth rotation is the most immersive, but can cause motion sickness.
     */
    Smooth = 2
}
/** List of string keys for {@link RotationType}. */
export declare const RotationTypeNames: ("None" | "Snap" | "Smooth")[];
/**
 * Simpler character controller for smooth locomotin.
 *
 * It needs to have a physx component attached to it. And be on the root of the
 * player hierarchy.
 *
 * @see {@link PlayerControllerInput} for handling input from various input providers
 */
export declare class PlayerController extends Component {
    static TypeName: string;
    static onRegister(engine: WonderlandEngine): void;
    /** Tracked space onto which the rotation is applied */
    trackedSpace: Object3D;
    /**
     * Object containing the {@link PlayerControllerInput} to read inputs from.
     *
     * @note If not provided, this component will create a default one.
     */
    inputObject: Object3D | null;
    /** Walk speed multiplier. */
    walkSpeed: number;
    transformType: RotationType;
    /**
     * The number of degrees at which the player rotates when snap-rotating
     */
    snapDegrees: number;
    /**
     * Rotation speed multiplier.
     *
     * @note Only used when {@link transformType} is {@link RotationType.Smooth}.
     */
    rotationSpeed: number;
    /**
     * Input to feed the controller.
     *
     * Using a non-component input is valid, e.g.,
     *
     * ```ts
     * import {PlayerControllerInput} fromn '@wonderlandengine/interaction';
     *
     * class Input implements PlayerControllerInput {
     *     getRotationAxis(out: vec3) {
     *         out[0] = 1.0;
     *     }
     * }
     *
     * controller.input = new Input();
     * ```
     */
    input: PlayerControllerInput;
    private _activeCamera;
    private _physx;
    private _snapped;
    /** @override */
    start(): void;
    /** @override */
    onActivate(): void;
    /** @override */
    update(dt: number): void;
    /**
     * Moves the player in the given direction.
     * @param movement The direction to move in.
     */
    move(movement: vec3): void;
    rotate(angle: number): void;
}
