import { quat, vec3 } from 'gl-matrix';
/** Constants. */
const StackSize = 4;
/** Temporaries. */
const _vectorA = vec3.create();
/**
 * Angular and linear velocities history tracker.
 *
 * This helper allows to accumulate and smooth the velocities
 * over multiple frames.
 */
export class HistoryTracker {
    /* Private Attributes. */
    /** List of linear velocities.  */
    _linear = new Array(StackSize);
    /** List of angular velocities.  */
    _angular = new Array(StackSize);
    /** Current position in the ring buffer.  */
    _curr = -1;
    /** Previous world space position of the object. */
    _previousPosition = vec3.create();
    /** Previous world space rotation of the object. */
    _previousRotation = quat.create();
    constructor() {
        for (let i = 0; i < StackSize; ++i) {
            this._linear[i] = vec3.create();
            this._angular[i] = vec3.create();
        }
    }
    /**
     * Update the history with the given object.
     *
     * @param target The target object to update from.
     * @param delta The delta time.
     */
    update(target, delta) {
        this._curr = (this._curr + 1) % StackSize;
        const linearOutput = this._linear[this._curr];
        const angularOutput = this._angular[this._curr];
        this._updateLinear(linearOutput, target, delta);
        this._updateAngular(angularOutput, target, delta);
    }
    /**
     * Update the history with the given [XR pose](https://developer.mozilla.org/en-US/docs/Web/API/XRPose).
     *
     * @remarks
     * Use this when available, because the velocities from the XR Pose are more accurate.
     *
     * @param xrPose The XR pose.
     * @param target The object to get the velocity from, in case the XR pose doesn't expose any.
     * @param delta The delta time.
     */
    updateFromPose(xrPose, space, target, delta) {
        /* eslint-disable */
        // @ts-ignore Unfortunately, typings are outdated.
        const velocity = xrPose.linearVelocity;
        // @ts-ignore Unfortunately, typings are outdated.
        const angular = xrPose.angularVelocity;
        /* eslint-enable */
        this._curr = (this._curr + 1) % StackSize;
        const linearOutput = this._linear[this._curr];
        if (velocity) {
            linearOutput[0] = velocity.x;
            linearOutput[1] = velocity.y;
            linearOutput[2] = velocity.z;
            space.transformVectorWorld(linearOutput);
        }
        else {
            this._updateLinear(linearOutput, target, delta);
        }
        const angularOutput = this._angular[this._curr];
        if (angular) {
            angularOutput[0] = angular.x;
            angularOutput[1] = angular.y;
            angularOutput[2] = angular.z;
            space.transformVectorWorld(angularOutput);
        }
        else {
            this._updateAngular(angularOutput, target, delta);
        }
    }
    /**
     * Resets the history tracker.
     *
     * @remarks
     * This method needs a target because it resets the history based on
     * the position of the target.
     *
     * @param target The object that was tracked.
     */
    reset(target) {
        for (const v of this._linear) {
            vec3.zero(v);
        }
        for (const v of this._angular) {
            vec3.zero(v);
        }
        this._curr = -1;
        const position = target.getPositionWorld(_vectorA);
        vec3.copy(this._previousPosition, position);
    }
    /**
     * Computes the linear velocity based on the current history.
     *
     * @remarks
     * This method isn't a simple getter and will perform computations,
     * Only use this once per frame or after the object is moved.
     *
     * @param out The output velocity.
     * @returns The `out` parameter.
     */
    velocity(out = vec3.create()) {
        vec3.zero(out);
        const count = this._linear.length;
        for (let i = 0; i < count; ++i) {
            vec3.add(out, out, this._linear[i]);
        }
        vec3.scale(out, out, 1.0 / count);
        return out;
    }
    /**
     * Computes the angular velocity based on the current history.
     *
     * @remarks
     * This method isn't a simple getter and will perform computations,
     * Only use once per frame or after the object is rotated.
     *
     * @param out The output angular velocity.
     * @returns vec3 The `out` parameter.
     */
    angular(out) {
        vec3.zero(out);
        const count = this._angular.length;
        for (let i = 0; i < count; ++i) {
            vec3.add(out, out, this._angular[i]);
        }
        vec3.scale(out, out, 1.0 / count);
        return out;
    }
    _updateLinear(out, target, delta) {
        const position = target.getPositionWorld(_vectorA);
        vec3.subtract(out, position, this._previousPosition);
        vec3.scale(out, out, 1.0 / delta);
        vec3.copy(this._previousPosition, position);
    }
    /** @ignore - Method not implemented */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _updateAngular(out, target, delta) {
        /* @todo: Handle angular velocity. */
        // const worldRot = target.getRotationWorld(quat.create());
        // const deltaRot = quatDelta(quat.create(), this._previousRotation, worldRot);
    }
}
