/// <reference types="webxr" />
import { Object3D } from '@wonderlandengine/api';
import { vec3 } from 'gl-matrix';
/**
 * Angular and linear velocities history tracker.
 *
 * This helper allows to accumulate and smooth the velocities
 * over multiple frames.
 */
export declare class HistoryTracker {
    /** List of linear velocities.  */
    private readonly _linear;
    /** List of angular velocities.  */
    private readonly _angular;
    /** Current position in the ring buffer.  */
    private _curr;
    /** Previous world space position of the object. */
    private _previousPosition;
    /** Previous world space rotation of the object. */
    private _previousRotation;
    constructor();
    /**
     * Update the history with the given object.
     *
     * @param target The target object to update from.
     * @param delta The delta time.
     */
    update(target: Object3D, delta: number): void;
    /**
     * Update the history with the given [XR pose](https://developer.mozilla.org/en-US/docs/Web/API/XRPose).
     *
     * @remarks
     * Use this when available, because the velocities from the XR Pose are more accurate.
     *
     * @param xrPose The XR pose.
     * @param target The object to get the velocity from, in case the XR pose doesn't expose any.
     * @param delta The delta time.
     */
    updateFromPose(xrPose: XRPose, space: Object3D, target: Object3D, delta: number): void;
    /**
     * Resets the history tracker.
     *
     * @remarks
     * This method needs a target because it resets the history based on
     * the position of the target.
     *
     * @param target The object that was tracked.
     */
    reset(target: Object3D): void;
    /**
     * Computes the linear velocity based on the current history.
     *
     * @remarks
     * This method isn't a simple getter and will perform computations,
     * Only use this once per frame or after the object is moved.
     *
     * @param out The output velocity.
     * @returns The `out` parameter.
     */
    velocity(out?: vec3): vec3;
    /**
     * Computes the angular velocity based on the current history.
     *
     * @remarks
     * This method isn't a simple getter and will perform computations,
     * Only use once per frame or after the object is rotated.
     *
     * @param out The output angular velocity.
     * @returns vec3 The `out` parameter.
     */
    angular(out: vec3): vec3;
    private _updateLinear;
    /** @ignore - Method not implemented */
    private _updateAngular;
}
