import { Component, Emitter } from '@wonderlandengine/api';
import { ARProvider } from './AR-provider.js';
export interface TrackingStatusEvent {
}
/** Position returned by a SLAM provider's hit-test against real-world surfaces. */
export interface HitTestResult {
    position: {
        x: number;
        y: number;
        z: number;
    };
}
/**
 * AR cameras will carry a tracking mode (SLAM, Face tracking, image tracking, etc)
 */
export interface ITrackingMode {
    readonly component: Component;
    update?: (delta: number) => void;
    /**
     * Optional camera pose output as a world-space dual quaternion (8 floats).
     *
     * When provided, camera components (e.g. `ar-slam-camera`) may apply it via
     * `object.setTransformWorld(...)`.
     */
    getCameraTransformWorld?: () => ArrayLike<number> | null;
    /**
     * Optional camera projection matrix (16 floats).
     *
     * When provided, camera components (e.g. `ar-slam-camera`) may apply it via
     * `view.projectionMatrix.set(...)`.
     *
     * @param out The array to write the projection matrix to.
     * @returns `true` if the projection matrix was written, `false` otherwise.
     */
    getCameraProjectionMatrix?: (out: Float32Array) => boolean;
    init?(features?: string[]): void;
    startSession: () => void;
    endSession: () => void;
    /**
     * Optional: set up a hit-test source against real-world surfaces.
     *
     * Called once when an AR session starts (e.g. to request an
     * `XRHitTestSource` for WebXR, or to enable the Zappar `WorldTracker`).
     * Providers that do not require explicit setup may leave this unimplemented.
     */
    setupHitTest?(): Promise<void>;
    /**
     * Optional: tear down any hit-test resources created by {@link setupHitTest}.
     *
     * Called when the AR session ends.
     */
    teardownHitTest?(): void;
    /**
     * Optional: return the current hit-test result for a ray cast from the
     * centre of the camera into the real world.
     *
     * Returns `null` when no surface has been found this frame.
     */
    getHitTestResult?(): HitTestResult | null;
}
export declare abstract class TrackingMode implements ITrackingMode {
    readonly component: Component;
    readonly provider: ARProvider;
    constructor(provider: ARProvider, component: Component);
    abstract startSession(): void;
    abstract endSession(): void;
}
/**
 * Face attachment point enum
 *
 * Values are sorted by position on the face from top to bottom,
 * as these will appear in the Wonderland Editor in this order.
 */
export declare enum FaceAttachmentPoint {
    Forehead = "forehead",
    EyeOuterCornerLeft = "eye outer corner left",
    EyeOuterCornerRight = "eye outer corner right",
    EyeBrowInnerLeft = "eyebrow inner left",
    EyeBrowInnerRight = "eyebrow inner right",
    EyeBrowCenterLeft = "eyebrow center left",
    EyeBrowCenterRight = "eyebrow center right",
    EyeBrowOuterLeft = "eyebrow outer left",
    EyeBrowOuterRight = "eyebrow outer right",
    EarLeft = "ear left",
    EarRight = "ear right",
    EyeLeft = "eye left",
    EyeRight = "eye right",
    NoseBridge = "nose bridge",
    NoseTip = "nose tip",
    CheekLeft = "cheek left",
    CheekRight = "cheek right",
    Mouth = "mouth",
    MouthCornerLeft = "mouth corner left",
    MouthCornerRight = "mouth corner right",
    UpperLip = "upper lip",
    LowerLip = "lower lip",
    Chin = "chin"
}
export interface FaceLoadingEvent {
    maxDetections: number;
    pointsPerDetection: number;
    indices: [{
        a: number;
        b: number;
        c: number;
    }];
    uvs: [{
        u: number;
        v: number;
    }];
}
export interface FaceFoundEvent {
    id: number;
    vertices: [{
        x: number;
        y: number;
        z: number;
    }];
    normals: [{
        x: number;
        y: number;
        z: number;
    }];
    attachmentPoints: {
        [value in FaceAttachmentPoint]: {
            position: {
                x: number;
                y: number;
                z: number;
            };
        };
    };
    transform: {
        position: {
            x: number;
            y: number;
            z: number;
        };
        rotation: {
            x: number;
            y: number;
            z: number;
            w: number;
        };
        scale: number;
        scaledWidth: number;
        scaledHeight: number;
        scaledDepth: number;
    };
}
export interface FaceLostEvent {
    id: number;
}
export interface FaceTrackingMode extends ITrackingMode {
    readonly onFaceScanning: Emitter<[event: FaceLoadingEvent]>;
    readonly onFaceLoading: Emitter<[event: FaceLoadingEvent]>;
    readonly onFaceFound: Emitter<[event: FaceFoundEvent]>;
    readonly onFaceUpdate: Emitter<[event: FaceFoundEvent]>;
    readonly onFaceLost: Emitter<[event: FaceLostEvent]>;
}
export interface VPSMeshFoundEvent {
    id: string;
    position: {
        x: number;
        y: number;
        z: number;
    };
    rotation: {
        x: number;
        y: number;
        z: number;
        w: number;
    };
    geometry: {
        index: {
            array: Uint32Array;
            itemSize: 1;
        };
        attributes: [
            {
                name: 'position';
                array: Float32Array;
                itemSize: 3;
            },
            {
                name: 'color';
                array: Float32Array;
                itemSize: 3;
            }
        ];
    };
}
export interface VPSMeshUpdatedEvent {
    id: string;
    position: {
        x: number;
        y: number;
        z: number;
    };
    rotation: {
        x: number;
        y: number;
        z: number;
        w: number;
    };
}
export interface VPSMeshLostEvent {
    id: string;
}
export interface VPSWayPointEvent {
    name: string;
    position: {
        x: number;
        y: number;
        z: number;
    };
    rotation: {
        x: number;
        y: number;
        z: number;
        w: number;
    };
}
export interface VPSTrackingMode extends ITrackingMode {
    readonly onMeshFound: Emitter<[event: VPSMeshFoundEvent]>;
    readonly onWaySpotFound: Emitter<[event: VPSWayPointEvent]>;
    readonly onWaySpotUpdated: Emitter<[event: VPSWayPointEvent]>;
    readonly onWaySpotLost: Emitter<[event: VPSWayPointEvent]>;
}
export interface ImageScanningEvent {
    imageTargets: {
        /**
         * Detected image name
         */
        name: string;
        type: 'flat' | 'cylindrical' | 'conical';
        metadata: any | null;
        geometry: {
            arcLengthRadians?: number;
            arcStartRadians?: number;
            height?: number;
            radiusBottom?: number;
            radiusTop?: number;
            scaleWidth?: number;
            scaledHeight?: number;
        };
        properties?: {
            height: number;
            width: number;
            isRotated: boolean;
            left: number;
            top: number;
            moveable: boolean;
            originalHeight: number;
            originalWidth: number;
            physicalWidthInMeters: number | null;
        } | null;
    }[];
}
export interface ImageTrackedEvent {
    name: string;
    position: {
        x: number;
        y: number;
        z: number;
    };
    rotation: {
        x: number;
        y: number;
        z: number;
        w: number;
    };
    scale: number;
    scaleWidth: number;
    scaledHeight: number;
    type: 'flat' | 'cylindrical' | 'conical';
    height?: number;
    radiusTop?: number;
    radiusBottom?: number;
    arcStartRadians?: number;
    arcLengthRadians?: number;
}
export interface ImageTrackingMode extends ITrackingMode {
    readonly onImageScanning: Emitter<[event: ImageScanningEvent]>;
    readonly onImageFound: Emitter<[event: ImageTrackedEvent]>;
    readonly onImageUpdate: Emitter<[event: ImageTrackedEvent]>;
    readonly onImageLost: Emitter<[event: ImageTrackedEvent]>;
    /**
     * Pre-registers an image tracking target so the provider can start
     * detecting it.
     *
     * Providers that do not require pre-registration (e.g. WebXR mind-ar)
     * may leave this unimplemented.
     *
     * @param source URL or binary data for the target file (e.g. a Zappar `.zpt` path).
     * @param options.name Must match the `imageId` used by tracker components on this target.
     * @param options.physicalWidthInMeters Optional physical width for physically-correct scale.
     */
    registerTarget?(source: string | ArrayBuffer, options: {
        name: string;
        physicalWidthInMeters?: number;
    }): Promise<void>;
}
