import { Emitter, RetainEmitter, WonderlandEngine, Component } from '@wonderlandengine/api';
import { ARProvider } from './AR-provider.js';
import { FaceTrackingMode, ImageTrackingMode, ITrackingMode } from './tracking-mode.js';
import { TrackingType } from './tracking-type.js';
/**
 * ARSession
 *
 * - Manages all AR sessions.
 * - Registers dependencies (i.e., providers)
 * - Handles global callbacks when AR sessions are started or ended
 */
export declare class ARSession {
    private static engines;
    /** A tracking provider is a library that provides tracking capabilities, e.g.,
     * WebXR Device API, 8th Wall, mind-ar-js, etc.
     */
    private _trackingProviders;
    /** Current running provider when AR session is running */
    private _currentTrackingProvider;
    /** Emits and event when the AR session is ready */
    readonly onARSessionReady: RetainEmitter;
    /** Emits and event when an AR session was started */
    readonly onSessionStart: RetainEmitter<[trackingProvider: ARProvider]>;
    /** Emits and event when an AR session was ended */
    readonly onSessionEnd: Emitter<[trackingProvider: ARProvider]>;
    private _engine;
    private _sceneHasLoaded;
    private _arSessionIsReady;
    private _registeredCameras;
    private _readyCameras;
    /** Wonderland Engine instance this AR session is running on */
    get engine(): WonderlandEngine;
    /**
     * @returns a shallow copy of all registered providers
     */
    get registeredProviders(): ReadonlyArray<ARProvider>;
    /**
     * Retrieve the first registered provider that supports the given tracking type.
     * This mirrors the provider selection logic used by {@link getTrackingProvider}.
     */
    getPreferredARProvider(type: TrackingType): ARProvider | null;
    /** Convenience helper for UI decisions (e.g. whether an AR start button is needed). */
    supportsInstantTracking(type: TrackingType): boolean;
    /**
     * Retrieve tracking implementation for given type.
     *
     * @returns The tracking instance or `null` if no provider was
     *     able to provide given tracking type
     */
    getTrackingProvider(type: TrackingType.Face, component: Component): FaceTrackingMode;
    getTrackingProvider(type: TrackingType.Image, component: Component): ImageTrackingMode;
    getTrackingProvider(type: TrackingType, component: Component): ITrackingMode;
    /** Get registered {@link ARProvider} based on {@link ARProvider#name}. */
    getARProviderByName(name: string): ARProvider | null;
    /**
     * Get or create an AR session attached to given engine
     *
     * @param engine The engine to retrieve the AR session for.
     * @returns The current AR session, or creates one if none exists.
     */
    static getSessionForEngine(engine: WonderlandEngine): ARSession;
    /**
     * Called by AR camera components during initialization to participate in
     * AR session readiness gating.
     */
    registerARCameraComponent(): void;
    /**
     * Called by AR camera components once they completed their startup path
     * and are ready to accept `startSession` calls.
     */
    markARCameraReady(): void;
    private constructor();
    /**
     * Registers tracking provider.
     *
     * Makes sure it is loaded and hooks into providers onSessionStart,
     * onSessionLoaded events.
     */
    registerTrackingProvider(provider: ARProvider): Promise<void>;
    private checkProviderLoadProgress;
    private onWLSceneLoaded;
    /** Stop a running AR session (if any) */
    stopARSession(): void;
    private onProviderSessionStart;
    private onProviderSessionEnd;
}
