/**
 * Hit-test reticle backed by Zappar plane anchors from `WorldTracker`
 * (requires `@zappar/zappar >= 4.x`).
 *
 * Each frame the component casts a ray from the centre of `camera` into the
 * set of planes detected by {@link ZapparProvider}'s `WorldTracker` and moves
 * the owning object to the closest intersection point.  A {@link MeshComponent}
 * on the same object is shown while a valid hit is found and hidden otherwise.
 *
 * This is the Zappar equivalent of:
 * - `hit-test-location-root`  (WebXR Device API hit-test)
 * - `hit-test-location-xr8`   (8th Wall SLAM + XY-plane intersection)
 *
 * **Setup**
 * 1. Attach this component to the reticle object (which should have a
 *    {@link MeshComponent} for visual feedback).
 * 2. Set the `camera` property to the scene object that carries
 *    {@link ARSLAMCamera}.
 * 3. The `SpawnMeshOnReticle` component works alongside this component –
 *    tapping the screen (or selecting on WebXR) will spawn content at this
 *    object's position.
 *
 * **Coordinate spaces**
 * Zappar plane poses are expressed in Zappar world space.  The WLE camera
 * object's transform is driven directly from the Zappar camera pose matrix
 * (via `WorldTracking_Zappar`), so `camera.getPositionWorld()` and
 * `camera.getForwardWorld()` are already in the same coordinate space as the
 * plane poses.  No additional transform is required.
 */
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
import { Component, MeshComponent } from '@wonderlandengine/api';
import { property } from '@wonderlandengine/api/decorators.js';
import { vec3 } from 'gl-matrix';
import { ARSession } from '@wonderlandengine/ar-tracking';
import { ZapparProvider } from './zappar-provider.js';
export class HitTestLocationZappar extends Component {
    static TypeName = 'hit-test-location-zappar';
    /** The scene object that carries the {@link ARSLAMCamera} component. */
    camera;
    _provider = null;
    _mesh = null;
    // Pre-allocated scratch buffers to avoid GC pressure in the update loop.
    _camPos = new Float32Array(3);
    _camFwd = new Float32Array(3);
    _planeNormal = vec3.create();
    _planeOrigin = vec3.create();
    _toOrigin = vec3.create();
    _hitPoint = vec3.create();
    start() {
        const arSession = ARSession.getSessionForEngine(this.engine);
        arSession.onSessionStart.add(this._onSessionStart);
        arSession.onSessionEnd.add(this._onSessionEnd);
        this._mesh = this.object.getComponent(MeshComponent) ?? null;
        if (this._mesh)
            this._mesh.active = false;
        // Deactivate update loop until a valid Zappar session starts.
        this.active = false;
    }
    update() {
        const provider = this._provider;
        if (!provider)
            return;
        if (!this.camera) {
            console.warn(`[${this.type}] 'camera' property is not set.`);
            return;
        }
        const worldTracker = provider.worldTracker;
        if (!worldTracker || worldTracker.planes.size === 0) {
            if (this._mesh)
                this._mesh.active = false;
            return;
        }
        const cameraPose = provider.slamCameraPoseMatrix;
        if (!cameraPose) {
            if (this._mesh)
                this._mesh.active = false;
            return;
        }
        // -------------------------------------------------------------------
        // Ray definition in WLE / Zappar world space.
        // The camera's world transform has already been set from the Zappar
        // camera pose so these values are directly comparable to plane poses.
        // -------------------------------------------------------------------
        this.camera.getPositionWorld(this._camPos);
        this.camera.getForwardWorld(this._camFwd);
        let closestT = Infinity;
        let hitFound = false;
        for (const plane of worldTracker.planes.values()) {
            // plane.pose() returns a column-major 4×4 matrix in Zappar world space.
            const planeMat = plane.pose(cameraPose);
            // Column 1 = local Y-axis = surface normal for horizontal planes.
            this._planeNormal[0] = planeMat[4];
            this._planeNormal[1] = planeMat[5];
            this._planeNormal[2] = planeMat[6];
            // Column 3 = translation = a point on the plane.
            this._planeOrigin[0] = planeMat[12];
            this._planeOrigin[1] = planeMat[13];
            this._planeOrigin[2] = planeMat[14];
            // Ray–plane intersection: t = dot(planeOrigin - camPos, N) / dot(D, N)
            const denom = vec3.dot(this._camFwd, this._planeNormal);
            // Skip planes that are nearly parallel to the ray.
            if (Math.abs(denom) < 1e-6)
                continue;
            vec3.sub(this._toOrigin, this._planeOrigin, this._camPos);
            const t = vec3.dot(this._toOrigin, this._planeNormal) / denom;
            // Only accept intersections in front of the camera and closer than
            // the best so far.
            if (t > 1e-3 && t < closestT) {
                closestT = t;
                vec3.scaleAndAdd(this._hitPoint, this._camPos, this._camFwd, t);
                hitFound = true;
            }
        }
        if (hitFound) {
            this.object.setPositionWorld(this._hitPoint);
            if (this._mesh)
                this._mesh.active = true;
        }
        else {
            if (this._mesh)
                this._mesh.active = false;
        }
    }
    _onSessionStart = (provider) => {
        if (!(provider instanceof ZapparProvider))
            return;
        this._provider = provider;
        const wt = provider.enableWorldTracker();
        if (!wt) {
            // @zappar/zappar < 4.x or WorldTracker unavailable – degrade
            // gracefully rather than throwing.
            console.warn(`[${this.type}] Zappar WorldTracker (plane anchors) is not` +
                ' available in the installed SDK version. Upgrade' +
                ' @zappar/zappar to >= 4.x to enable plane-based hit tests.');
            return;
        }
        this.active = true;
    };
    _onSessionEnd = (provider) => {
        if (provider !== this._provider)
            return;
        this._provider = null;
        this.active = false;
        if (this._mesh)
            this._mesh.active = false;
    };
}
__decorate([
    property.object()
], HitTestLocationZappar.prototype, "camera", void 0);
