import { TrackingMode } from '@wonderlandengine/ar-tracking';
/**
 * Implementation of SLAM (World Tracking) based on the WebXR Device API
 *
 * Depends on WEBXR_REQUIRED_FEATURES, WEBXR_OPTIONAL_FEATURES global variables.
 *
 * TODO: change this when it's moved to auto constants.
 */
export class WorldTracking_WebXR extends TrackingMode {
    _config;
    _hitTestSource = null;
    constructor(provider, component, config) {
        super(provider, component);
        this._config = config;
    }
    startSession() {
        this.provider.startSession(this._config.requiredFeatures, this._config.optionalFeatures);
    }
    endSession() {
        this.provider.endSession();
    }
    async setupHitTest() {
        const provider = this.provider;
        const session = provider.xrSession;
        if (!session) {
            console.warn('[WorldTracking_WebXR] setupHitTest called without an active XR session.');
            return;
        }
        if (!session.requestHitTestSource) {
            console.warn('[WorldTracking_WebXR] Hit-test feature is not available. ' +
                "Add 'hit-test' to your WebXR optional features.");
            return;
        }
        const engine = this.provider.engine;
        const viewerSpace = engine.xr?.referenceSpaceForType('viewer');
        if (!viewerSpace) {
            console.warn('[WorldTracking_WebXR] Viewer reference space not available.');
            return;
        }
        this._hitTestSource =
            (await session.requestHitTestSource({ space: viewerSpace })) ?? null;
    }
    teardownHitTest() {
        this._hitTestSource?.cancel();
        this._hitTestSource = null;
    }
    getHitTestResult() {
        if (!this._hitTestSource)
            return null;
        const engine = this.provider.engine;
        const frame = engine.xr?.frame;
        if (!frame)
            return null;
        const results = frame.getHitTestResults(this._hitTestSource);
        if (results.length === 0)
            return null;
        const viewerSpace = engine.xr.referenceSpaceForType('viewer');
        if (!viewerSpace)
            return null;
        const pose = results[0].getPose(viewerSpace);
        if (!pose)
            return null;
        // Hit-test result is in viewer space; transform to world space via the
        // camera object's current world transform.
        const worldPos = this.component.object.transformPointWorld(new Array(3), [
            pose.transform.position.x,
            pose.transform.position.y,
            pose.transform.position.z,
        ]);
        return { position: { x: worldPos[0], y: worldPos[1], z: worldPos[2] } };
    }
}
