import { quat, quat2, vec3 } from 'gl-matrix';
/**
 * Convert degrees to radians.
 *
 * @param degrees Value, in **degrees**.
 */
export function toRad(degrees) {
    return (degrees * Math.PI) / 180.0;
}
/**
 * Convert radians to degrees.
 *
 * @param degrees Value, in **radians**.
 */
export function toDegree(radians) {
    return (radians * 180) / Math.PI;
}
/**
 * Check whether two points are **almost** equal.
 *
 * @param a Source point.
 * @param b Target point.
 * @param epsilon Epsilon threshold.
 * @returns `true` if almost equal, `false` otherwise.
 */
export function isPointEqual(a, b, epsilon) {
    return (Math.abs(a[0] - b[0]) < epsilon &&
        Math.abs(a[1] - b[1]) < epsilon &&
        Math.abs(a[2] - b[2]) < epsilon);
}
/**
 * Compute the relative transformation from source to target.
 *
 * @note
 * - The result transform is expressed in the target's space.
 * - In order to get the world transform of `source`, the target's world
 *   transform will need to be multiplied by the result of this function.
 *
 * @param out The destination.
 * @param source The source object.
 * @param target The target object.
 * @returns The `out` parameter.
 */
export const computeRelativeTransform = (function () {
    const _rotationA = quat.create();
    const _rotationB = quat.create();
    const _pointA = vec3.create();
    const _pointB = vec3.create();
    return function (out, source, target) {
        const handToLocal = quat2.getReal(_rotationA, target);
        quat.invert(handToLocal, handToLocal);
        /* Transform rotation into target's space */
        const rot = quat2.getReal(_rotationB, source);
        quat.multiply(rot, handToLocal, rot);
        quat.normalize(rot, rot);
        /* Transform position into target's space */
        const position = quat2.getTranslation(_pointA, source);
        const handPosition = quat2.getTranslation(_pointB, target);
        vec3.sub(position, position, handPosition);
        vec3.transformQuat(position, position, handToLocal);
        return quat2.fromRotationTranslation(out, rot, position);
    };
})();
