import { Component, Object3D } from '@wonderlandengine/api';
import { vec3 } from 'gl-matrix';
/**
 * Typename for a `PlayerControllerInput` component.
 *
 * Use this to create your own input component that can
 * be retrieved by {@link PlayerController}:
 *
 * ```ts
 * import {PlayerControllerInput, PlayerControllerInputTypename} fromn '@wonderlandengine/interaction';
 *
 * export class CustomPlayerInput implements PlayerControllerInput {
 *     static TypeName = PlayerControllerInputTypename;
 *     ...
 * }
 * ```
 */
export declare const PlayerControllerInputTypename = "player-controller-input";
/**
 * Defines an interface for bridging various input methods into a unified control scheme.
 *
 * The `PlayerControllerInput` serves as an abstraction layer that enables an application to
 * process input data from different sources such as keyboards, game controllers,
 * and VR controllers in a consistent manner. Implementations of this interface should
 * provide mechanisms to interpret and translate these diverse inputs into a set
 * of standardized movement and rotation vectors that the application can easily use
 * for navigation and interaction within a 3D environment.
 *
 * @example
 * ```js
 * export class CustomInput extends Component implements PlayerControllerInput {
 *   static TypeName = InputBridgeTypename;
 *
 *   \/** @override *\/
 *   getRotationAxis(out: vec3) {
 *       // Always rotate by 10 degrees around the. You can input your own values here,
 *       // coming from a keyboard, the network, etc...
 *       out[1] = 10;
 *       return out;
 *   }
 *
 *   \/** @override *\/
 *   getMovementAxis(out: vec3) {
 *       // Always move to the right. You can input your own values here,
 *       // coming from a keyboard, the network, etc...
 *       out[0] = 1;
 *       return out;
 *   }
 *}
 *```
 */
export interface PlayerControllerInput extends Component {
    /**
     * Get the rotation axis.
     *
     * @param out The destination array.
     * @returns The `out` parameter;
     */
    getRotationAxis(out: vec3): vec3;
    /**
     * Get the movement axis.
     *
     * @param out The destination array.
     * @returns The `out` parameter;
     */
    getMovementAxis(out: vec3): vec3;
}
/**
 * A default implementation of that handles input from:
 * - Keyboard
 * - VR controller
 */
export declare class DefaultPlayerControllerInput extends Component implements PlayerControllerInput {
    static TypeName: string;
    keyboard: boolean;
    vr: boolean;
    leftControlObject: Object3D;
    rightControlObject: Object3D;
    deadzoneThreshold: number;
    private _keyPress;
    private _inputLeft;
    private _inputRight;
    private _direction;
    private _rotation;
    /** @override */
    onActivate(): void;
    /** @override */
    onDeactivate(): void;
    /** @override */
    update(): void;
    /** @override */
    getRotationAxis(out: vec3): vec3;
    /** @override */
    getMovementAxis(out: vec3): vec3;
    /** @hidden */
    private _onKeyPressed;
    /** @hidden */
    private _onKeyReleased;
}
