var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
import { vec3 } from 'gl-matrix';
import { CollisionComponent, CollisionEventType, Component, InputComponent, MeshComponent, PhysXComponent, } from '@wonderlandengine/api';
import { property } from '@wonderlandengine/api/decorators.js';
import { Grabbable } from './grabbable.js';
import { GrabSearchMode, GrabPoint, InteractorVisualState, InteractorVisualStateNames, } from './grab-point.js';
import { componentError, setComponentsActive } from '../utils/wle.js';
import { DefaultInteractorInput } from './interactor-input.js';
/**
 * Manages interaction capabilities of a VR controller or a similar input device.
 *
 * The `Interactor` class enables an entity to grip or interact with objects that
 * implement the {@link Interactable} interface.
 */
export class Interactor extends Component {
    static TypeName = 'interactor';
    static onRegister(engine) {
        engine.registerComponent(DefaultInteractorInput);
    }
    /** Properties */
    inputObject;
    meshRoot = null;
    visualStateOnGrab = InteractorVisualState.Visible;
    trackedSpace = null;
    /** Private Attributes. */
    _input;
    /** Collision component of this object. */
    _collision = null;
    /** Physx component of this object. */
    _physx = null;
    /**
     * Physx collision callback index.
     *
     * @hidden
     */
    _physxCallback = null;
    /** Cached interactable after it's gripped. */
    _grabbable = null;
    _onGrabStart = () => {
        this.checkForNearbyInteractables();
    };
    _onGrabEnd = () => {
        this.stopInteraction();
    };
    #currentlyCollidingWith = null;
    /**
     * Set the collision component needed to perform
     * grab interaction
     *
     * @param collision The collision component
     *
     * @returns This instance, for chaining
     */
    start() {
        this._collision = this.object.getComponent(CollisionComponent, 0);
        this._physx = this.object.getComponent(PhysXComponent, 0);
        if (!this._collision && !this._physx) {
            throw new Error('grabber.start(): No collision or physx component found');
        }
        let maybeInput = (this.inputObject ?? this.object).getComponent(DefaultInteractorInput);
        if (!maybeInput) {
            const search = (object) => {
                const input = object.getComponent(InputComponent);
                if (input)
                    return input;
                return object.parent ? search(object.parent) : null;
            };
            const input = search(this.object);
            if (!input) {
                throw new Error(componentError(this, "'InteractoInput' component not provided, and no native input component could found"));
            }
            maybeInput = this.object.addComponent(DefaultInteractorInput, {
                inputObject: input.object,
            });
        }
        this._input = maybeInput;
    }
    onActivate() {
        if (!this._collision) {
            this._physxCallback = this._physx.onCollision(this.onPhysxCollision);
        }
        this._input.onGrabStart.add(this._onGrabStart);
        this._input.onGrabEnd.add(this._onGrabEnd);
    }
    onDeactivate() {
        if (this._physxCallback !== null) {
            this._physx.removeCollisionCallback(this._physxCallback);
            this._physxCallback = null;
        }
        if (!this._input.isDestroyed) {
            this._input.onGrabStart.remove(this._onGrabStart);
            this._input.onGrabEnd.remove(this._onGrabEnd);
        }
    }
    /**
     * Force this interactor to start interacting with the given interactable.
     *
     * @param interactable The interactable to process.
     */
    startInteraction(interactable, handleId) {
        const handle = interactable.grabPoints[handleId];
        if (handle.interactor) {
            if (!handle.transferable)
                return;
            interactable.release(handle.interactor);
        }
        handle._interactor = this;
        this._grabbable = interactable;
        interactable.grab(this, handleId);
        let hidden = this.visualStateOnGrab === InteractorVisualState.Hidden;
        if (interactable.interactorVisualState !== InteractorVisualState.None) {
            hidden = interactable.interactorVisualState === InteractorVisualState.Hidden;
        }
        if (handle.interactorVisualState !== InteractorVisualState.None) {
            hidden = handle.interactorVisualState === InteractorVisualState.Hidden;
        }
        if (this.meshRoot && hidden) {
            setComponentsActive(this.meshRoot, false, MeshComponent);
        }
    }
    /**
     * Check for nearby interactable, and notifies one if this interactor
     * interacts with it.
     */
    checkForNearbyInteractables() {
        // TODO: Allocation.
        const thisPosition = this.object.getPositionWorld();
        let minDistance = Number.POSITIVE_INFINITY;
        let grabbableId = null;
        let handleId = null;
        /* Prioritize collision / physx over distance grab */
        // TODO: Link from handle to grabbable would be better.
        // Just need to ensure than a handle can't be referenced by
        // 2 grabbables.
        const overlaps = this._collision ? this._collision.queryOverlaps() : null;
        let overlapHandle = null;
        if (overlaps) {
            /** @todo: The API should instead allow to check for overlap on given objects. */
            const overlaps = this._collision.queryOverlaps();
            for (const overlap of overlaps) {
                overlapHandle = overlap.object.getComponent(GrabPoint);
                if (overlapHandle)
                    break;
            }
        }
        if (!overlapHandle && this.#currentlyCollidingWith) {
            /** @todo: The API should instead allow to check for overlap on given objects. */
            overlapHandle = this.#currentlyCollidingWith;
        }
        /** @todo: Optimize with a typed list of handle, an octree? */
        const grabbables = this.scene.getActiveComponents(Grabbable);
        for (let i = 0; i < grabbables.length; ++i) {
            const grabbable = grabbables[i];
            for (let h = 0; h < grabbable.grabPoints.length; ++h) {
                const handle = grabbable.grabPoints[h];
                let dist = Number.POSITIVE_INFINITY;
                switch (handle.searchMode) {
                    case GrabSearchMode.Distance: {
                        /** @todo: Add interactor grab origin */
                        const otherPosition = handle.object.getPositionWorld();
                        dist = vec3.sqrDist(thisPosition, otherPosition);
                        break;
                    }
                    case GrabSearchMode.Overlap: {
                        dist = overlapHandle === handle ? 0.0 : dist;
                        break;
                    }
                }
                const maxDistanceSq = handle.maxDistance * handle.maxDistance;
                if (dist < maxDistanceSq && dist < minDistance) {
                    minDistance = dist;
                    grabbableId = i;
                    handleId = h;
                }
            }
        }
        if (grabbableId !== null) {
            this.startInteraction(grabbables[grabbableId], handleId);
        }
    }
    onPhysxCollision = (type, other) => {
        const grab = other.object.getComponent(GrabPoint);
        if (!grab)
            return;
        if (type == CollisionEventType.TriggerTouch) {
            this.#currentlyCollidingWith = grab;
        }
        else if (grab === this.#currentlyCollidingWith) {
            this.#currentlyCollidingWith = null;
        }
    };
    /**
     * Force this interactor to stop interacting with the
     * currently bound interactable.
     */
    stopInteraction() {
        if (this._grabbable && !this._grabbable.isDestroyed) {
            this._grabbable.release(this);
        }
        this._grabbable = null;
        if (this.meshRoot && !this.meshRoot.isDestroyed) {
            setComponentsActive(this.meshRoot, true, MeshComponent);
        }
    }
    /**
     * Current interactable handled by this interactor. If no interaction is ongoing,
     * this getter returns `null`.
     */
    get interactable() {
        return this._grabbable;
    }
    /** {@link InteractorInput} */
    get input() {
        return this._input;
    }
}
__decorate([
    property.object()
], Interactor.prototype, "inputObject", void 0);
__decorate([
    property.object()
], Interactor.prototype, "meshRoot", void 0);
__decorate([
    property.enum(InteractorVisualStateNames, InteractorVisualState.Visible)
], Interactor.prototype, "visualStateOnGrab", void 0);
__decorate([
    property.object({ required: true })
], Interactor.prototype, "trackedSpace", void 0);
