import { world_tracker_quality_t as WorldTrackerQuality, plane_orientation_t as PlaneOrientation, anchor_status_t as AnchorStatus, } from '@zappar/zappar-cv';
import { z } from './zappar.js';
import { Event, Event1 } from './event.js';
export { world_tracker_quality_t as WorldTrackerQuality, transform_orientation_t as TransformOrientation, plane_orientation_t as PlaneOrientation, anchor_status_t as AnchorStatus, world_scale_mode_t as WorldScaleMode, } from '@zappar/zappar-cv';
/**
 * Attaches content to a point on a surface in front of the user as it moves around in the camera view.
 * @see https://docs.zap.works/universal-ar/javascript/instant-world-tracking/
 */
export class WorldTracker {
    /**
     * Constructs a new WorldTracker.
     * @param _pipeline - The pipeline that this tracker will operate within.
     */
    constructor(pipeline) {
        this.onQuality = new Event1();
        this.onInitializing = new Event();
        this.onQualityGood = new Event();
        this.onQualityLimited = new Event();
        this.quality = WorldTrackerQuality.WORLD_TRACKER_QUALITY_INITIALIZING;
        this.worldAnchor = {
            id: 'world',
            poseCameraRelative: mirror => this._z.world_tracker_world_anchor_pose_camera_relative(this._impl, mirror !== null && mirror !== void 0 ? mirror : false),
            pose: (cameraPose, mirror) => this._z.world_tracker_world_anchor_pose(this._impl, cameraPose, mirror !== null && mirror !== void 0 ? mirror : false),
            status: AnchorStatus.ANCHOR_STATUS_INITIALIZING,
        };
        this.groundAnchor = {
            id: 'ground',
            poseCameraRelative: mirror => this._z.world_tracker_ground_anchor_pose_camera_relative(this._impl, mirror !== null && mirror !== void 0 ? mirror : false),
            pose: (cameraPose, mirror) => this._z.world_tracker_ground_anchor_pose(this._impl, cameraPose, mirror !== null && mirror !== void 0 ? mirror : false),
            status: AnchorStatus.ANCHOR_STATUS_INITIALIZING,
        };
        this.planes = new Map();
        this.customAnchors = new Map();
        this.resetIfLimitedForMS = 5000;
        this._hadFirstPlane = false;
        this._frameUpdate = () => {
            this.worldAnchor.status = this._z.world_tracker_world_anchor_status(this._impl);
            this.worldAnchor.id = this._z.world_tracker_world_anchor_id(this._impl);
            this.groundAnchor.status = this._z.world_tracker_ground_anchor_status(this._impl);
            this.groundAnchor.id = this._z.world_tracker_ground_anchor_id(this._impl);
            const presentAnchors = new Set();
            const planeCount = this._z.world_tracker_plane_anchor_count(this._impl);
            for (let i = 0; i < planeCount; i++) {
                const id = this._z.world_tracker_plane_anchor_id(this._impl, i);
                presentAnchors.add(id);
                let anchor = this.planes.get(id);
                if (!anchor) {
                    anchor = {
                        id,
                        status: AnchorStatus.ANCHOR_STATUS_INITIALIZING,
                        poseCameraRelative: mirror => this._z.world_tracker_plane_anchor_pose_camera_relative(this._impl, i, mirror !== null && mirror !== void 0 ? mirror : false),
                        pose: (cameraPose, mirror) => this._z.world_tracker_plane_anchor_pose(this._impl, i, cameraPose, mirror !== null && mirror !== void 0 ? mirror : false),
                        polygon: new Float32Array(),
                        polygonVersion: 0,
                        orientation: PlaneOrientation.PLANE_ORIENTATION_HORIZONTAL,
                        firstPlane: !this._hadFirstPlane,
                    };
                    this._hadFirstPlane = true;
                    this.planes.set(id, anchor);
                }
                anchor.poseCameraRelative = mirror => this._z.world_tracker_plane_anchor_pose_camera_relative(this._impl, i, mirror !== null && mirror !== void 0 ? mirror : false);
                anchor.pose = (cameraPose, mirror) => this._z.world_tracker_plane_anchor_pose(this._impl, i, cameraPose, mirror !== null && mirror !== void 0 ? mirror : false);
                anchor.status = this._z.world_tracker_plane_anchor_status(this._impl, i);
                anchor.polygon = this._z.world_tracker_plane_anchor_polygon_data(this._impl, i);
                anchor.polygonVersion = this._z.world_tracker_plane_anchor_polygon_version(this._impl, i);
                anchor.orientation = this._z.world_tracker_plane_anchor_orientation(this._impl, i);
            }
            for (const anchor of this.planes.values()) {
                if (!presentAnchors.has(anchor.id)) {
                    if (anchor.firstPlane)
                        this._hadFirstPlane = false;
                    this.planes.delete(anchor.id);
                }
            }
            const lastQuality = this.quality;
            this.quality = this._z.world_tracker_quality(this._impl);
            if (lastQuality !== this.quality) {
                if (this._limitedTimeout && this.quality !== WorldTrackerQuality.WORLD_TRACKER_QUALITY_LIMITED) {
                    window.clearTimeout(this._limitedTimeout);
                    this._limitedTimeout = undefined;
                }
                switch (this.quality) {
                    case WorldTrackerQuality.WORLD_TRACKER_QUALITY_GOOD:
                        this.onQualityGood.emit();
                        break;
                    case WorldTrackerQuality.WORLD_TRACKER_QUALITY_INITIALIZING:
                        this.onInitializing.emit();
                        break;
                    case WorldTrackerQuality.WORLD_TRACKER_QUALITY_LIMITED:
                        if (!this._limitedTimeout && this.resetIfLimitedForMS > 0) {
                            this._limitedTimeout = window.setTimeout(() => {
                                this.reset();
                                this._limitedTimeout = undefined;
                            }, this.resetIfLimitedForMS);
                        }
                        this.onQualityLimited.emit();
                        break;
                }
                this.onQuality.emit(this.quality);
            }
        };
        this._z = z();
        this._pipeline = pipeline;
        this._pipeline._onFrameUpdateInternal.bind(this._frameUpdate);
        this._impl = this._z.world_tracker_create(this._pipeline._getImpl());
        this.worldAnchor.id = this._z.world_tracker_world_anchor_id(this._impl);
        this.groundAnchor.id = this._z.world_tracker_ground_anchor_id(this._impl);
    }
    /**
     * Destroys the world tracker.
     */
    destroy() {
        this._pipeline._onFrameUpdateInternal.unbind(this._frameUpdate);
        this._z.world_tracker_destroy(this._impl);
    }
    reset() {
        this._z.world_tracker_reset(this._impl);
    }
    /**
     * Gets/sets the enabled state of the world tracker.
     * Disable when not in use to save computational resources during frame processing.
     */
    get enabled() {
        return this._z.world_tracker_enabled(this._impl);
    }
    set enabled(e) {
        this._z.world_tracker_enabled_set(this._impl, e);
    }
    get horizontalPlaneDetectionEnabled() {
        return this._z.world_tracker_horizontal_plane_detection_enabled(this._impl);
    }
    set horizontalPlaneDetectionEnabled(v) {
        this._z.world_tracker_horizontal_plane_detection_enabled_set(this._impl, v);
    }
    get verticalPlaneDetectionEnabled() {
        return this._z.world_tracker_vertical_plane_detection_enabled(this._impl);
    }
    set verticalPlaneDetectionEnabled(v) {
        this._z.world_tracker_vertical_plane_detection_enabled_set(this._impl, v);
    }
    get verticalPlaneDetectionSupported() {
        return this._z.world_tracker_vertical_plane_detection_supported(this._impl);
    }
    get tracksDataEnabled() {
        return this._z.world_tracker_tracks_data_enabled(this._impl);
    }
    set tracksDataEnabled(v) {
        this._z.world_tracker_tracks_data_enabled_set(this._impl, v);
    }
    get projectionsDataEnabled() {
        return this._z.world_tracker_projections_data_enabled(this._impl);
    }
    get tracksData() {
        return this._z.world_tracker_tracks_data(this._impl);
    }
    get projectionsData() {
        return this._z.world_tracker_projections_data(this._impl);
    }
    get scaleMode() {
        return this._z.world_tracker_scale_mode(this._impl);
    }
    set scaleMode(mode) {
        this._z.world_tracker_scale_mode_set(this._impl, mode);
    }
    computePointsDataMatrix(screenWidth, screenHeight, mirror = false) {
        return this._z.world_tracker_points_data_matrix(this._impl, screenWidth, screenHeight, mirror);
    }
}
