import { z } from './zappar.js';
/**
 * The permissions that may be requested.
 */
export var Permission;
(function (Permission) {
    /**
     * Permission to access camera images.
     */
    Permission[Permission["CAMERA"] = 0] = "CAMERA";
    /**
     * Permission to access device motion data (e.g. accelerometer and gyro). Some tracking algorithms require this data to operate.
     */
    Permission[Permission["MOTION"] = 1] = "MOTION";
})(Permission || (Permission = {}));
/**
 * Checks if the browser has currently granted relevant permissions.
 * @param onlyPermsission - The exclusive permission to query, otherwise all are queried.
 * @returns The permission granted state. 'true' if permission is granted.
 */
export function permissionGranted(onlyPermsission) {
    switch (onlyPermsission) {
        case Permission.CAMERA:
            return z().permission_granted_camera();
        case Permission.MOTION:
            return z().permission_granted_motion();
        default:
            return z().permission_granted_all();
    }
}
/**
 * Checks if the browser has currently denied relevant permissions.
 * @param onlyPermsission - The exclusive permission to query, otherwise all are queried.
 * @returns The permission granted state. 'true' if permission is denied.
 */
export function permissionDenied(onlyPermission) {
    switch (onlyPermission) {
        case Permission.CAMERA:
            return z().permission_denied_camera();
        case Permission.MOTION:
            return z().permission_denied_motion();
        default:
            return z().permission_denied_any();
    }
}
/**
 * Requests the browser to grant relevant permissions.
 *
 * This may or may not trigger a browser-provided user dialog prompting a permission choice.
 *
 * @param onlyPermission - The exclusive permission to query, otherwise all are requested.
 * @returns A Promise containing granted status. 'true' if granted.
 */
export function permissionRequest(onlyPermission) {
    switch (onlyPermission) {
        case Permission.CAMERA:
            z().permission_request_camera();
            break;
        case Permission.MOTION:
            z().permission_request_motion();
            break;
        default:
            z().permission_request_all();
            break;
    }
    // eslint-disable-next-line no-async-promise-executor
    return new Promise(async (resolve) => {
        // eslint-disable-next-line no-constant-condition
        while (true) {
            await _nextFrame();
            switch (onlyPermission) {
                case Permission.CAMERA:
                    if (z().permission_granted_camera()) {
                        resolve(true);
                        return;
                    }
                    if (z().permission_denied_camera()) {
                        resolve(false);
                        return;
                    }
                    break;
                case Permission.MOTION:
                    if (z().permission_granted_motion()) {
                        resolve(true);
                        return;
                    }
                    if (z().permission_denied_motion()) {
                        resolve(false);
                        return;
                    }
                    break;
                default:
                    if (z().permission_granted_camera() && z().permission_granted_motion()) {
                        resolve(true);
                        return;
                    }
                    if (z().permission_denied_camera() || z().permission_denied_motion()) {
                        resolve(false);
                        return;
                    }
                    break;
            }
        }
    });
}
/**
 * Shows Zappar's built-in UI to request camera and motion data permissions
 * @returns A promise containing granted status.
 */
export async function permissionRequestUI(options) {
    return await z().permission_request_ui_promise(options);
}
/**
 * Shows Zappar's built-in permission denied UI.
 */
export function permissionDeniedUI(options) {
    return z().permission_denied_ui(options);
}
function _nextFrame() {
    return new Promise(resolve => requestAnimationFrame(() => resolve()));
}
