var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { anchor_status_t, world_tracker_quality_t, camera_profile_t } from "./gen/zappar";
import { profile } from "./profile";
import { Pipeline } from "./pipeline";
import { Source } from "./source";
import { cameraRotationForScreenOrientation } from "./cameramodel";
import { zcout } from "./loglevel";
import { deleteCameraSource } from "./camera-source-map";
import { mat4, vec3, quat } from "gl-matrix";
import { ImageBlitGl } from "./image-blit-gl";
import { permissionRequestUI } from "./permission";
const _circleBoundary = new Float32Array(32 * 2);
for (let i = 0; i < 32; i++) {
    const angle = (i / 32) * 2 * Math.PI;
    _circleBoundary[i * 2] = Math.cos(angle) * 0.5;
    _circleBoundary[i * 2 + 1] = Math.sin(angle) * 0.5;
}
const _quat = quat.create();
const _trans = vec3.create();
const _up = vec3.fromValues(0, 1, 0);
const _ground = mat4.create();
export class WebXRCameraSource extends Source {
    constructor(_impl, _pipeline) {
        super();
        this._impl = _impl;
        this._pipeline = _pipeline;
        this.sessionNumber = 0;
        this._isPaused = true;
        this._xrSession = null;
        this._xrLocalSpace = null;
        this._xrViewerSpace = null;
        this._xrGlBinding = null;
        this._xrWebGLLayer = null;
        this._offscreenCanvas = new OffscreenCanvas(640, 480);
        this._latestFrame = null;
        this._latestWtData = null;
        this._wtOptions = {
            horizontalPlaneDetection: true,
            verticalPlaneDetection: false
        };
        this._lastFrameTime = -1;
        this._cameraToDeviceTransform = mat4.create();
        this._cameraModel = new Float32Array([300, 300, 160, 120, 0, 0]);
        this._rawCameraModel = new Float32Array([300, 300, 160, 120, 0, 0]);
        this._profile = camera_profile_t.DEFAULT;
        this._permissionRequestShown = false;
        this._planeHitTestSource = null;
        this._customAnchorSetRequests = new Map();
        this._customAnchors = new Map();
        this._onXRFrame = (time, frame) => {
            var _a, _b, _c, _d, _e, _f;
            if (this._isPaused)
                return;
            (_a = this._xrSession) === null || _a === void 0 ? void 0 : _a.requestAnimationFrame(this._onXRFrame);
            if (!this._xrLocalSpace || !this._xrGlBinding || !this._xrViewerSpace)
                return;
            let viewerPose = frame.getViewerPose(this._xrViewerSpace);
            if (!viewerPose)
                return;
            let camView;
            let camera;
            let texture;
            for (const view of viewerPose.views) {
                camera = view.camera;
                if (camera) {
                    texture = this._xrGlBinding.getCameraImage(camera);
                    if (texture) {
                        camView = view;
                        break;
                    }
                }
            }
            if (!texture)
                return;
            const viewport = (_b = this._xrWebGLLayer) === null || _b === void 0 ? void 0 : _b.getViewport(camView);
            this._updateCameraModel(camView, viewport);
            const localPose = frame.getViewerPose(this._xrLocalSpace);
            const gl = this._offscreenContext;
            let blitWidth = camera.width;
            let blitHeight = camera.height;
            let blitMax = Math.max(blitWidth, blitHeight);
            if (blitMax > 1280) {
                blitWidth = Math.round(1280 * blitWidth / blitMax);
                blitHeight = Math.round(1280 * blitHeight / blitMax);
            }
            if (blitWidth != this._offscreenCanvas.width || blitHeight != this._offscreenCanvas.height) {
                this._offscreenCanvas.width = blitWidth;
                this._offscreenCanvas.height = blitHeight;
                gl.viewport(0, 0, this._offscreenCanvas.width, this._offscreenCanvas.height);
            }
            // Blit the texture to the framebuffer
            gl.bindTexture(gl.TEXTURE_2D, texture);
            this._imageBlitGl.doBlit();
            gl.bindTexture(gl.TEXTURE_2D, null);
            if (this._latestFrame)
                this._latestFrame.close();
            this._latestFrame = this._offscreenCanvas.transferToImageBitmap();
            let rot = cameraRotationForScreenOrientation(false);
            const rotateMatrix = (m, rot) => {
                let ret = m.slice();
                if (rot === 90) {
                    ret[0] = m[1];
                    ret[4] = m[5];
                    ret[8] = m[9];
                    ret[12] = m[13];
                    ret[1] = -m[0];
                    ret[5] = -m[4];
                    ret[9] = -m[8];
                    ret[13] = -m[12];
                }
                if (rot === 180) {
                    ret[0] = -m[0];
                    ret[4] = -m[4];
                    ret[8] = -m[8];
                    ret[12] = -m[12];
                    ret[1] = -m[1];
                    ret[5] = -m[5];
                    ret[9] = -m[9];
                    ret[13] = -m[13];
                }
                if (rot === 270) {
                    ret[0] = -m[1];
                    ret[4] = -m[5];
                    ret[8] = -m[9];
                    ret[12] = -m[13];
                    ret[1] = m[0];
                    ret[5] = m[4];
                    ret[9] = m[8];
                    ret[13] = m[12];
                }
                return ret;
            };
            for (const [customAnchor, req] of this._customAnchorSetRequests.entries()) {
                if (req.delay > 0) {
                    req.delay--;
                    continue;
                }
                this._customAnchorSetRequests.delete(customAnchor);
                mat4.getRotation(_quat, req.pose);
                mat4.getTranslation(_trans, req.pose);
                const rigid = new XRRigidTransform({ x: _trans[0], y: _trans[1], z: _trans[2], w: 1 }, { x: _quat[0], y: _quat[1], z: _quat[2], w: _quat[3] });
                (_d = (_c = frame.createAnchor) === null || _c === void 0 ? void 0 : _c.call(frame, rigid, this._xrLocalSpace)) === null || _d === void 0 ? void 0 : _d.then(anchor => {
                    const existing = this._customAnchors.get(customAnchor);
                    if (existing)
                        existing.anchor.delete();
                    this._customAnchors.set(customAnchor, { anchor, version: req.version });
                });
            }
            const plane_anchors = [];
            if (this._planeHitTestSource && this._wtOptions.horizontalPlaneDetection && this._xrViewerSpace) {
                const hitTestResults = frame.getHitTestResults(this._planeHitTestSource);
                for (const hitTestResult of hitTestResults) {
                    const worldPose = hitTestResult.getPose(this._xrLocalSpace);
                    const poseInCamera = hitTestResult.getPose(this._xrViewerSpace);
                    if (!worldPose || !poseInCamera)
                        continue;
                    if (this._isPlaneVertical(worldPose)) {
                        if (this._wtOptions.verticalPlaneDetection) {
                            const xzRotation = [1, 0];
                            // Looking to compute a 2D rotation matrix to rotate the X and Z coordinates
                            // of the WebXR plane such that the Z coordinate points down (ie aligned
                            // with -Y in the world).
                            // [a -b]
                            // [b  a]
                            // Dot product of X and Z axis directions with world up direction
                            const xDp = worldPose.transform.matrix[1];
                            const zDp = worldPose.transform.matrix[9];
                            const normScale = 1.0 / Math.sqrt(xDp * xDp + zDp * zDp);
                            xzRotation[0] = -zDp * normScale;
                            xzRotation[1] = -xDp * normScale;
                            const cp = poseInCamera.transform.matrix.slice();
                            const m0 = cp[0];
                            const m1 = cp[1];
                            const m2 = cp[2];
                            const m8 = cp[8];
                            const m9 = cp[9];
                            const m10 = cp[10];
                            const a = xzRotation[0];
                            const b = -xzRotation[1]; // -b gives us the inverse
                            cp[0] = a * m0 + b * m8;
                            cp[1] = a * m1 + b * m9;
                            cp[2] = a * m2 + b * m10;
                            cp[8] = -b * m0 + a * m8;
                            cp[9] = -b * m1 + a * m9;
                            cp[10] = -b * m2 + a * m10;
                            plane_anchors.push({
                                id: "plane-" + plane_anchors.length,
                                horizontal: false,
                                status: anchor_status_t.ANCHOR_STATUS_TRACKING,
                                pose: rotateMatrix(cp, rot),
                                boundary: _circleBoundary,
                                boundaryVersion: 0
                            });
                            break;
                        }
                        continue; // skip vertical planes if not requested
                    }
                    if (poseInCamera) {
                        // Update ground height to the lowest detected plane
                        if (localPose) {
                            const planeY = worldPose.transform.matrix[13];
                            const viewerHeight = localPose.transform.matrix[13] - planeY;
                            if (viewerHeight > 0.8 && viewerHeight < 1.8) {
                                if (this._groundH === undefined || planeY < this._groundH) {
                                    this._groundH = planeY;
                                }
                            }
                        }
                        plane_anchors.push({
                            id: "plane-" + plane_anchors.length,
                            horizontal: true,
                            status: anchor_status_t.ANCHOR_STATUS_TRACKING,
                            pose: rotateMatrix(poseInCamera.transform.matrix, rot),
                            boundary: _circleBoundary,
                            boundaryVersion: 0
                        });
                        break;
                    }
                }
            }
            const custom_anchors = [];
            if (this._xrViewerSpace && frame.trackedAnchors) {
                for (const [o, anchor] of this._customAnchors.entries()) {
                    const pose = frame.getPose(anchor.anchor.anchorSpace, this._xrViewerSpace);
                    if (!pose)
                        continue;
                    custom_anchors.push({
                        id: o,
                        status: frame.trackedAnchors.has(anchor.anchor) ? anchor_status_t.ANCHOR_STATUS_TRACKING : anchor_status_t.ANCHOR_STATUS_PAUSED,
                        pose: rotateMatrix((_e = pose === null || pose === void 0 ? void 0 : pose.transform) === null || _e === void 0 ? void 0 : _e.matrix, rot),
                        poseVersion: anchor.version
                    });
                }
            }
            if (!localPose) {
                if (!this._latestWtData)
                    return;
                this._latestWtData = Object.assign(Object.assign({}, this._latestWtData), { world_anchor_status: anchor_status_t.ANCHOR_STATUS_PAUSED, ground_anchor_status: anchor_status_t.ANCHOR_STATUS_PAUSED, quality: world_tracker_quality_t.WORLD_TRACKER_QUALITY_LIMITED });
            }
            else {
                const world_pose = rotateMatrix(localPose.transform.inverse.matrix, rot);
                let ground_pose = mat4.create();
                mat4.fromTranslation(_ground, [0, (_f = this._groundH) !== null && _f !== void 0 ? _f : -1.5, 0]);
                mat4.multiply(ground_pose, localPose.transform.inverse.matrix, _ground);
                ground_pose = rotateMatrix(ground_pose, rot);
                this._latestWtData = {
                    world_anchor_status: anchor_status_t.ANCHOR_STATUS_TRACKING,
                    world_anchor_pose: world_pose,
                    ground_anchor_pose: ground_pose,
                    ground_anchor_status: this._groundH === undefined ? anchor_status_t.ANCHOR_STATUS_INITIALIZING : anchor_status_t.ANCHOR_STATUS_TRACKING,
                    plane_anchors,
                    custom_anchors,
                    quality: world_tracker_quality_t.WORLD_TRACKER_QUALITY_GOOD,
                };
            }
        };
        this._onXREnd = (ev) => {
            this._xrLocalSpace = null;
            this._xrSession = null;
            this._latestFrame = null;
            this._latestWtData = null;
            this._customAnchorSetRequests.clear();
            this._customAnchors.clear();
            this._planeHitTestSource = null;
            this._groundH = undefined;
            if (this._isPaused)
                return;
            if (this._permissionRequestShown)
                return;
            this._permissionRequestShown = true;
            permissionRequestUI({ ignoreUserActivation: true }).then(() => {
                this._permissionRequestShown = false;
                this._syncCamera();
            });
        };
        this._hasStartedOrientation = false;
        this._lastTimestamp = -1;
        this._deviceMotionListener = (ev) => {
            let pipeline = Pipeline.get(this._pipeline);
            if (!pipeline)
                return;
            let timeStamp = (ev.timeStamp !== undefined && ev.timeStamp !== null) ? ev.timeStamp : performance.now();
            let interval = ev.interval;
            if (this._lastTimestamp > -1 && !profile.trustSensorIntervals) {
                interval = Math.max(timeStamp - this._lastTimestamp, interval);
            }
            this._lastTimestamp = timeStamp;
            interval *= profile.intervalMultiplier;
            if (ev.acceleration !== null &&
                ev.acceleration.x !== null &&
                ev.acceleration.y !== null &&
                ev.acceleration.z !== null) {
                pipeline.motionAccelerometerWithoutGravitySubmitInt(timeStamp, interval, ev.acceleration.x * profile.deviceMotionMutliplier, ev.acceleration.y * profile.deviceMotionMutliplier, ev.acceleration.z * profile.deviceMotionMutliplier);
            }
            if (ev.accelerationIncludingGravity !== null &&
                ev.accelerationIncludingGravity.x !== null &&
                ev.accelerationIncludingGravity.y !== null &&
                ev.accelerationIncludingGravity.z !== null) {
                pipeline.motionAccelerometerSubmit(timeStamp, ev.accelerationIncludingGravity.x * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.y * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.z * profile.deviceMotionMutliplier);
                pipeline.motionAccelerometerWithGravitySubmitInt(timeStamp, interval, ev.accelerationIncludingGravity.x * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.y * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.z * profile.deviceMotionMutliplier);
            }
            if (ev.rotationRate !== null &&
                ev.rotationRate.alpha !== null &&
                ev.rotationRate.beta !== null &&
                ev.rotationRate.gamma !== null && this._hasStartedOrientation) {
                pipeline.motionRotationRateSubmit(timeStamp, ev.rotationRate.alpha * Math.PI / -180.0, ev.rotationRate.beta * Math.PI / -180.0, ev.rotationRate.gamma * Math.PI / -180.0);
                pipeline.motionRotationRateSubmitInt(timeStamp, interval, ev.rotationRate.alpha, ev.rotationRate.beta, ev.rotationRate.gamma);
            }
            else if (!this._hasStartedOrientation) {
                this._startDeviceOrientation();
            }
        };
        zcout("Using WebXR camera source");
        let gl = this._offscreenCanvas.getContext('webgl', { xrCompatible: true });
        if (!gl)
            throw new Error("Couldn't get a webgl context from OffscreenCanvas");
        this._offscreenContext = gl;
        this._imageBlitGl = new ImageBlitGl(gl, false);
    }
    static IsSupported() {
        var _a, _b, _c;
        // Core spec objects
        if (!(navigator.xr && ((_a = window.XRSession) === null || _a === void 0 ? void 0 : _a.prototype) && ((_b = window.XRFrame) === null || _b === void 0 ? void 0 : _b.prototype)))
            return false;
        // Check required WebXR features
        if (!((_c = window.XRCamera) === null || _c === void 0 ? void 0 : _c.prototype))
            return false; // raw-camera-access
        if (!('domOverlayState' in XRSession.prototype))
            return false; // dom-overlay
        if (!('requestHitTestSource' in XRSession.prototype))
            return false; // hit-test
        if (!('createAnchor' in XRFrame.prototype))
            return false; // anchors
        // Samsung Internet (at least v29) has all the raw-camera-access types
        // and functions but rejects sessions when it is a requiredFeature, so for now
        // we need to sniff user-agent to block it
        if (navigator.userAgent.indexOf(' SamsungBrowser/') > -1)
            return false;
        return true;
    }
    destroy() {
        this.pause();
        deleteCameraSource(this._impl);
    }
    setProfile(p) {
        this._profile = p;
    }
    _stop() {
        if (!this._xrSession)
            return;
        this._xrSession.end();
    }
    pause() {
        this._isPaused = true;
        let p = Pipeline.get(this._pipeline);
        if (p && p.currentCameraSource === this)
            p.currentCameraSource = undefined;
        this._stopDeviceMotion();
        this._syncCamera();
    }
    start() {
        var _a;
        let p = Pipeline.get(this._pipeline);
        if (p && p.currentCameraSource !== this) {
            (_a = p.currentCameraSource) === null || _a === void 0 ? void 0 : _a.pause();
            p.currentCameraSource = this;
        }
        this._isPaused = false;
        this._startDeviceMotion();
        this._syncCamera();
    }
    optionsUpdated(options) {
        this._wtOptions = options;
    }
    resetTracking() {
        var _a;
        (_a = this._xrSession) === null || _a === void 0 ? void 0 : _a.end();
        // TODO (remove anchors, reset local space)
    }
    getFrame(currentlyProcessing) {
        var _a, _b;
        if (!this._latestFrame)
            return;
        if (currentlyProcessing)
            return;
        let currentTime = performance.now();
        if (currentTime < (this._lastFrameTime + 25))
            return;
        this._lastFrameTime = currentTime;
        const imageBitmap = this._latestFrame;
        this._latestFrame = null;
        let rotation = cameraRotationForScreenOrientation(false);
        let pipeline = Pipeline.get(this._pipeline);
        if (!pipeline)
            return;
        let isPortrait = (rotation == 90) || (rotation == 270);
        let [dataWidth, dataHeight] = profile.getDataSize(this._profile);
        let ibWidth = isPortrait ? imageBitmap.height : imageBitmap.width;
        let ibHeight = isPortrait ? imageBitmap.width : imageBitmap.height;
        let cropHeight = Math.round(dataWidth * ibHeight / ibWidth);
        if (cropHeight < dataHeight) {
            dataHeight = cropHeight;
        }
        else {
            dataWidth = Math.round(dataHeight * ibWidth / ibHeight);
        }
        const rawX = isPortrait ? this._rawCameraModel[1] : this._rawCameraModel[0];
        const rawY = isPortrait ? this._rawCameraModel[0] : this._rawCameraModel[1];
        const rawU = isPortrait ? this._rawCameraModel[3] : this._rawCameraModel[2];
        const rawV = isPortrait ? this._rawCameraModel[2] : this._rawCameraModel[3];
        this._cameraModel[0] = 0.5 * rawX * dataWidth / rawU;
        this._cameraModel[1] = 0.5 * rawY * dataHeight / rawV;
        this._cameraModel[2] = dataWidth * 0.5;
        this._cameraModel[3] = dataHeight * 0.5;
        let token = pipeline.registerToken({
            dataWidth: imageBitmap.width,
            dataHeight: imageBitmap.height,
            texture: undefined,
            userFacing: false,
            cameraSource: this,
            cameraSourceData: { wtData: this._latestWtData },
            cameraModel: this._cameraModel.slice(),
            cameraToDevice: this._cameraToDeviceTransform
        });
        (_b = (_a = Pipeline.get(this._pipeline)) === null || _a === void 0 ? void 0 : _a.sendImageBitmapToWorker) === null || _b === void 0 ? void 0 : _b.call(_a, imageBitmap, rotation, false, token, this._cameraModel, this._cameraToDeviceTransform, this._profile);
        return;
    }
    _syncCamera() {
        var _a, _b, _c;
        return __awaiter(this, void 0, void 0, function* () {
            if (this._isPaused) {
                if (this._xrSession)
                    this._stop();
                return;
            }
            if (this._xrSession)
                return;
            let hasActivation = (_a = navigator.userActivation) === null || _a === void 0 ? void 0 : _a.isActive;
            if (!hasActivation) {
                if (this._permissionRequestShown)
                    return;
                this._permissionRequestShown = true;
                permissionRequestUI().then(() => {
                    this._permissionRequestShown = false;
                    this._syncCamera();
                });
                return;
            }
            if (navigator.xr) {
                try {
                    const session = yield navigator.xr.requestSession('immersive-ar', { requiredFeatures: ['dom-overlay', 'camera-access', 'hit-test', 'anchors'], domOverlay: { root: document.documentElement } });
                    const refSpace = yield session.requestReferenceSpace('local');
                    this._xrViewerSpace = yield session.requestReferenceSpace('viewer');
                    this._xrWebGLLayer = new XRWebGLLayer(session, this._offscreenContext);
                    session.updateRenderState({ baseLayer: this._xrWebGLLayer });
                    session.addEventListener('end', this._onXREnd);
                    session.requestAnimationFrame(this._onXRFrame);
                    this.sessionNumber++;
                    this._xrSession = session;
                    this._xrLocalSpace = refSpace;
                    this._xrGlBinding = new XRWebGLBinding(session, this._offscreenContext);
                    // Create a reference space that is a ray pointing down [0, -0.3] in normalized device coordinates
                    const offsetRay = new XRRay(new DOMPointReadOnly(0, 0, 0), { x: 0, y: -0.3, z: -1, w: 0 });
                    (_c = (_b = session.requestHitTestSource) === null || _b === void 0 ? void 0 : _b.call(session, {
                        space: this._xrViewerSpace,
                        offsetRay: offsetRay,
                        entityTypes: ['plane']
                    })) === null || _c === void 0 ? void 0 : _c.then((hitTestSource) => {
                        this._planeHitTestSource = hitTestSource;
                    });
                }
                catch (ex) {
                    console.log('Exception starting immersive-ar session');
                }
            }
        });
    }
    _startDeviceOrientation() {
        if (this._hasStartedOrientation)
            return;
        this._hasStartedOrientation = true;
        window.addEventListener("deviceorientation", (ev) => {
            let pipeline = Pipeline.get(this._pipeline);
            if (!pipeline)
                return;
            let timeStamp = (ev.timeStamp !== undefined && ev.timeStamp !== null) ? ev.timeStamp : performance.now();
            if (ev.alpha === null || ev.beta === null || ev.gamma === null)
                return;
            // pipeline.motionAttitudeSubmit(timeStamp, ev.alpha, ev.beta, ev.gamma);
            pipeline.motionAttitudeSubmitInt(timeStamp, 0, ev.alpha, ev.beta, ev.gamma);
        });
    }
    _startDeviceMotion() {
        window.addEventListener("devicemotion", this._deviceMotionListener, false);
    }
    _stopDeviceMotion() {
        window.removeEventListener("devicemotion", this._deviceMotionListener);
    }
    uploadGL(info) {
        const pipeline = Pipeline.get(this._pipeline);
        const gl = pipeline === null || pipeline === void 0 ? void 0 : pipeline.glContext;
        if (!info ||
            info.texture ||
            !info.frame ||
            !pipeline ||
            !gl)
            return;
        let texture = pipeline.getVideoTexture();
        if (!texture)
            return;
        gl.bindTexture(gl.TEXTURE_2D, texture);
        gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, false);
        gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, info.frame);
        gl.bindTexture(gl.TEXTURE_2D, null);
        info.texture = texture;
    }
    setCustomAnchorPose(o, version, pose) {
        return __awaiter(this, void 0, void 0, function* () {
            this._customAnchorSetRequests.set(o, { version, pose: pose.slice(), delay: 3 });
        });
    }
    deleteCustomAnchor(o) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            (_a = this._customAnchors.get(o)) === null || _a === void 0 ? void 0 : _a.anchor.delete();
        });
    }
    _updateCameraModel(camView, viewport) {
        const p = camView.projectionMatrix;
        const u0 = (1 - p[8]) * viewport.width / 2 + viewport.x;
        const v0 = (1 - p[9]) * viewport.height / 2 + viewport.y;
        // Focal lengths in pixels (these are equal for square pixels)
        const ax = viewport.width / 2 * p[0];
        const ay = viewport.height / 2 * p[5];
        this._rawCameraModel[0] = ax;
        this._rawCameraModel[1] = ay;
        // Principal point in pixels
        this._rawCameraModel[2] = u0;
        this._rawCameraModel[3] = v0;
    }
    _isPlaneVertical(pose) {
        _quat[0] = pose.transform.orientation.x;
        _quat[1] = pose.transform.orientation.y;
        _quat[2] = pose.transform.orientation.z;
        _quat[3] = pose.transform.orientation.w;
        _up[0] = 0;
        _up[1] = 1;
        _up[2] = 0;
        vec3.transformQuat(_up, _up, _quat);
        if (Math.abs(_up[1]) < 0.5)
            return true;
        return false;
    }
}
