import { compileShader, linkProgram } from "./shader";
let shader;
let vbo;
export function disposeDrawQuad() {
    shader = undefined;
    vbo = undefined;
}
function generate(gl) {
    if (vbo)
        return vbo;
    vbo = gl.createBuffer();
    if (!vbo)
        throw new Error("Unable to create buffer object");
    let coords = [
        -0.5, 0.5, 0,
        -0.5, -0.5, 0,
        0.5, 0.5, 0,
        0.5, -0.5, 0
    ];
    gl.bindBuffer(gl.ARRAY_BUFFER, vbo);
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(coords), gl.STATIC_DRAW);
    gl.bindBuffer(gl.ARRAY_BUFFER, null);
    return vbo;
}
export function drawQuad(gl, projectionMatrix, modelviewMatrix, color) {
    let shader = getShader(gl);
    let v = generate(gl);
    gl.useProgram(shader.prog);
    gl.uniformMatrix4fv(shader.unif_projection, false, projectionMatrix);
    gl.uniformMatrix4fv(shader.unif_modelView, false, modelviewMatrix);
    gl.uniform4fv(shader.unif_color, color);
    gl.bindBuffer(gl.ARRAY_BUFFER, v);
    gl.vertexAttribPointer(shader.attr_position, 3, gl.FLOAT, false, 3 * 4, 0);
    gl.enableVertexAttribArray(shader.attr_position);
    gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);
    gl.disableVertexAttribArray(shader.attr_position);
    gl.bindBuffer(gl.ARRAY_BUFFER, null);
}
let vertexShaderSrc = `
#ifndef GL_ES
#define highp
#define mediump
#define lowp
#endif

uniform mat4 projMatrix;
uniform mat4 modelViewMatrix;
attribute vec4 position;

void main()
{
    gl_Position = projMatrix * modelViewMatrix * position;
}`;
let fragmentShaderSrc = `
#define highp mediump
#ifdef GL_ES
    // define default precision for float, vec, mat.
    precision highp float;
#else
#define highp
#define mediump
#define lowp
#endif

uniform vec4 color;

void main()
{
    gl_FragColor = color;
}`;
function getShader(gl) {
    if (shader)
        return shader;
    let prog = gl.createProgram();
    if (!prog)
        throw new Error("Unable to create program");
    let vertexShader = compileShader(gl, gl.VERTEX_SHADER, vertexShaderSrc);
    let fragmentShader = compileShader(gl, gl.FRAGMENT_SHADER, fragmentShaderSrc);
    gl.attachShader(prog, vertexShader);
    gl.attachShader(prog, fragmentShader);
    linkProgram(gl, prog);
    let unif_projection = gl.getUniformLocation(prog, "projMatrix");
    if (!unif_projection)
        throw new Error("Unable to get uniform location projMatrix");
    let unif_modelView = gl.getUniformLocation(prog, "modelViewMatrix");
    if (!unif_modelView)
        throw new Error("Unable to get uniform location modelViewMatrix");
    let unif_color = gl.getUniformLocation(prog, "color");
    if (!unif_color)
        throw new Error("Unable to get uniform location color");
    shader = {
        prog,
        unif_modelView,
        unif_projection,
        unif_color,
        attr_position: gl.getAttribLocation(prog, "position"),
    };
    return shader;
}
