import { profile } from "./profile";
import { compileShader, linkProgram } from "./shader";
import { mat4 } from "gl-matrix";
let identity = mat4.create();
export class PointsDraw {
    constructor(_gl) {
        this._gl = _gl;
    }
    dispose() {
        if (this._vbo)
            this._gl.deleteBuffer(this._vbo);
        this._vbo = undefined;
        if (this._shader)
            this._gl.deleteProgram(this._shader.prog);
        this._shader = undefined;
    }
    _generate(gl, points) {
        if (!this._vbo)
            this._vbo = gl.createBuffer();
        if (!this._vbo)
            throw new Error("Unable to create buffer object");
        gl.bindBuffer(gl.ARRAY_BUFFER, this._vbo);
        gl.bufferData(gl.ARRAY_BUFFER, points, gl.DYNAMIC_DRAW);
        gl.bindBuffer(gl.ARRAY_BUFFER, null);
        return this._vbo;
    }
    drawPoints(screenWidth, screenHeight, dataWidth, dataHeight, points, mirror, color, pointSize) {
        if (points.length === 0)
            return; // points = new Float32Array([0, 1, 0, 1]);
        let gl = this._gl;
        const reenableDepthTest = gl.isEnabled(gl.DEPTH_TEST);
        const reenableScissorTest = gl.isEnabled(gl.SCISSOR_TEST);
        const reenableCullFace = gl.isEnabled(gl.CULL_FACE);
        const reenableStencilTest = gl.isEnabled(gl.STENCIL_TEST);
        const reenableBlend = gl.isEnabled(gl.BLEND);
        const previousBoundTexture = gl.getParameter(gl.TEXTURE_BINDING_2D);
        const previousBoundFramebuffer = gl.getParameter(gl.FRAMEBUFFER_BINDING);
        const previousProgram = gl.getParameter(gl.CURRENT_PROGRAM);
        const previousActiveTexture = gl.getParameter(gl.ACTIVE_TEXTURE);
        gl.disable(gl.DEPTH_TEST);
        gl.disable(gl.SCISSOR_TEST);
        gl.disable(gl.CULL_FACE);
        gl.disable(gl.STENCIL_TEST);
        gl.disable(gl.BLEND);
        let shader = this._getCameraShader(gl);
        let vbo = this._generate(gl, points);
        gl.activeTexture(gl.TEXTURE0);
        gl.useProgram(shader.prog);
        gl.uniform3f(shader.unif_color, color[0], color[1], color[2]);
        gl.uniform1f(shader.unif_pointSize, pointSize);
        gl.uniformMatrix4fv(shader.unif_skinTexTransform, false, getPointsDataMatrix(dataWidth, dataHeight, screenWidth, screenHeight, mirror));
        gl.bindTexture(gl.TEXTURE_2D, null);
        gl.bindBuffer(gl.ARRAY_BUFFER, vbo);
        gl.vertexAttribPointer(shader.attr_position, 2, gl.FLOAT, false, 0, 0);
        gl.enableVertexAttribArray(shader.attr_position);
        gl.drawArrays(gl.POINTS, 0, points.length / 2);
        gl.disableVertexAttribArray(shader.attr_position);
        gl.bindFramebuffer(gl.FRAMEBUFFER, previousBoundFramebuffer);
        gl.useProgram(previousProgram);
        gl.bindTexture(gl.TEXTURE_2D, previousBoundTexture);
        gl.activeTexture(previousActiveTexture);
        if (reenableBlend)
            gl.enable(gl.BLEND);
        if (reenableCullFace)
            gl.enable(gl.CULL_FACE);
        if (reenableDepthTest)
            gl.enable(gl.DEPTH_TEST);
        if (reenableScissorTest)
            gl.enable(gl.SCISSOR_TEST);
        if (reenableStencilTest)
            gl.enable(gl.STENCIL_TEST);
    }
    _getCameraShader(gl) {
        if (this._shader)
            return this._shader;
        let prog = gl.createProgram();
        if (!prog)
            throw new Error("Unable to create program");
        let vertexShader = compileShader(gl, gl.VERTEX_SHADER, vertexShaderSrc);
        let fragmentShader = compileShader(gl, gl.FRAGMENT_SHADER, fragmentShaderSrc);
        gl.attachShader(prog, vertexShader);
        gl.attachShader(prog, fragmentShader);
        linkProgram(gl, prog);
        let unif_skinTexTransform = gl.getUniformLocation(prog, "skinTexTransform");
        if (!unif_skinTexTransform)
            throw new Error("Unable to get uniform location skinTexTransform");
        let unif_color = gl.getUniformLocation(prog, "color");
        if (!unif_color)
            throw new Error("Unable to get uniform location color");
        let unif_pointSize = gl.getUniformLocation(prog, "pointSize");
        if (!unif_pointSize)
            throw new Error("Unable to get uniform location pointSize");
        this._shader = {
            prog,
            unif_skinTexTransform,
            unif_color,
            unif_pointSize,
            attr_position: gl.getAttribLocation(prog, "position"),
        };
        return this._shader;
    }
}
let vertexShaderSrc = `
#ifndef GL_ES
#define highp
#define mediump
#define lowp
#endif

attribute vec2 position;
uniform mat4 skinTexTransform;
uniform float pointSize;

void main()
{
    gl_Position = skinTexTransform * vec4(position, 0.0, 1.0);
    // gl_Position = vec4(position.x * 0.0, position.y * 0.0, 0.0, 1.0);
    gl_PointSize = pointSize;
}`;
let fragmentShaderSrc = `
#define highp mediump
#ifdef GL_ES
    // define default precision for float, vec, mat.
    precision highp float;
#else
#define highp
#define mediump
#define lowp
#endif

uniform vec3 color;

void main()
{
    gl_FragColor = vec4(color, 1.0);
}`;
function cameraRotationForScreenOrientation() {
    if (window.screen.orientation && !profile.forceWindowOrientation) {
        switch (window.screen.orientation.type) {
            case "portrait-primary":
                return 270;
            case "landscape-secondary":
                return 180;
            case "portrait-secondary":
                return 90;
            default:
                return 0;
        }
    }
    else if (window.orientation !== undefined) {
        switch (window.orientation) {
            case 0: return 270;
            case 90: return 0;
            case 180: return 90;
            case -90: return 180;
        }
    }
    return 0;
}
export function getPointsDataMatrix(frameWidth, frameHeight, screenWidth, screenHeight, mirror) {
    let ret = mat4.create();
    let trans = mat4.create();
    // Translate to centre UV coords
    mat4.fromTranslation(trans, [-0.5 * frameWidth, -0.5 * frameHeight, 0]);
    mat4.multiply(ret, trans, ret);
    mat4.fromScaling(trans, [2 / frameWidth, 2 / -frameHeight, 1]);
    mat4.multiply(ret, trans, ret);
    if (mirror) {
        mat4.fromScaling(trans, [-1, 1, 1]);
        mat4.multiply(ret, trans, ret);
    }
    let screenAspect = screenWidth > screenHeight ? (screenWidth / screenHeight) : (screenHeight / screenWidth);
    let frameAspect = frameWidth / frameHeight;
    mat4.fromScaling(trans, [1, screenAspect / frameAspect, 1]);
    mat4.multiply(ret, trans, ret);
    // Apply rotation back into ZCV's landscape space
    mat4.fromRotation(trans, cameraRotationForScreenOrientation() * Math.PI / 180.0, [0, 0, 1]);
    mat4.multiply(ret, trans, ret);
    return ret;
}
