var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { camera_profile_t } from "./gen/zappar";
import { profile } from "./profile";
import { Pipeline } from "./pipeline";
import { mat4 } from "gl-matrix";
import { Source } from "./source";
import { cameraRotationForScreenOrientation } from "./cameramodel";
import { zcout } from "./loglevel";
import { deleteCameraSource } from "./camera-source-map";
import { ArrayBufferFromString } from "./array-from-string";
import { UvLayout, YUVConversionGL } from "./yuv-conversion-gl";
import { AndroidBridgeMessageHandler } from "./android-bridge-message-handler";
import { getSharedTracer } from "./tracing/sharedtracer";
import { BridgedWorldTracker } from "./bridged-world-tracker";
import { hasBridgeArrayBufferHeader, parseBridgeArrayBuffer } from "./bridged-message-parser";
import { BridgedD3Tracker } from "./bridged-d3-tracker";
// Some no-op tracing helpers for concise code and good dead code removal in bundlers
let tracer = null;
let traceStart = (name) => { };
let traceEnd = () => { };
if (!(typeof Z_TRACING)) {
    tracer = getSharedTracer();
    traceStart = (name) => { tracer.start(name); };
    traceEnd = () => { tracer.end(); };
}
export class BridgedCameraSource extends Source {
    constructor(_impl, _pipeline, _deviceId) {
        var _a, _b;
        super();
        this._impl = _impl;
        this._pipeline = _pipeline;
        this._deviceId = _deviceId;
        this._isPaused = true;
        this._cameraToScreenRotation = 0;
        this._cameraToDeviceTransform = mat4.create();
        this._cameraModel = new Float32Array([1200, 1200, 639.5, 359.5, 0, 0]);
        this._profile = camera_profile_t.DEFAULT;
        this._textureSizes = new Map();
        this._decodeBuffers = [];
        this._options = {};
        this._isAndroid = false; // iOS resets on start, Android only on stop
        this._frameInFlight = false;
        this._anchorPoseFloat32 = new Float32Array(16);
        this._anchorPoseUint16 = new Uint16Array(this._anchorPoseFloat32.buffer);
        // Motion handlers
        this._lastTimestamp = -1;
        this._deviceMotionListener = (ev) => {
            let pipeline = Pipeline.get(this._pipeline);
            if (!pipeline)
                return;
            let timeStamp = (ev.timeStamp !== undefined && ev.timeStamp !== null) ? ev.timeStamp : performance.now();
            const interval = ev.interval * profile.intervalMultiplier;
            if (profile.trustSensorIntervals) {
                if (this._lastTimestamp === -1)
                    this._lastTimestamp = timeStamp;
                else {
                    this._lastTimestamp += interval;
                    timeStamp = this._lastTimestamp;
                }
            }
            if (ev.acceleration !== null &&
                ev.acceleration.x !== null &&
                ev.acceleration.y !== null &&
                ev.acceleration.z !== null) {
                pipeline.motionAccelerometerWithoutGravitySubmitInt(timeStamp, interval, ev.acceleration.x * profile.deviceMotionMutliplier, ev.acceleration.y * profile.deviceMotionMutliplier, ev.acceleration.z * profile.deviceMotionMutliplier);
            }
            if (ev.accelerationIncludingGravity !== null &&
                ev.accelerationIncludingGravity.x !== null &&
                ev.accelerationIncludingGravity.y !== null &&
                ev.accelerationIncludingGravity.z !== null) {
                pipeline.motionAccelerometerSubmit(timeStamp, ev.accelerationIncludingGravity.x * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.y * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.z * profile.deviceMotionMutliplier);
                pipeline.motionAccelerometerWithGravitySubmitInt(timeStamp, interval, ev.accelerationIncludingGravity.x * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.y * profile.deviceMotionMutliplier, ev.accelerationIncludingGravity.z * profile.deviceMotionMutliplier);
            }
            if (ev.rotationRate !== null &&
                ev.rotationRate.alpha !== null &&
                ev.rotationRate.beta !== null &&
                ev.rotationRate.gamma !== null) {
                pipeline.motionRotationRateSubmit(timeStamp, ev.rotationRate.alpha * Math.PI / -180.0, ev.rotationRate.beta * Math.PI / -180.0, ev.rotationRate.gamma * Math.PI / -180.0);
                pipeline.motionRotationRateSubmitInt(timeStamp, interval, ev.rotationRate.alpha, ev.rotationRate.beta, ev.rotationRate.gamma);
            }
        };
        this._deviceOrientationListener = (ev) => {
            let pipeline = Pipeline.get(this._pipeline);
            if (!pipeline)
                return;
            let timeStamp = (ev.timeStamp !== undefined && ev.timeStamp !== null) ? ev.timeStamp : performance.now();
            if (ev.alpha === null || ev.beta === null || ev.gamma === null)
                return;
            // pipeline.motionAttitudeSubmit(timeStamp, ev.alpha, ev.beta, ev.gamma);
            pipeline.motionAttitudeSubmitInt(timeStamp, 0, ev.alpha, ev.beta, ev.gamma);
        };
        if (AndroidBridgeMessageHandler.IsSupported()) {
            this._messageHandler = AndroidBridgeMessageHandler.SharedInstance();
            this._isAndroid = true;
        }
        else {
            this._messageHandler = (_b = (_a = window.webkit) === null || _a === void 0 ? void 0 : _a.messageHandlers) === null || _b === void 0 ? void 0 : _b.bridgedCamera;
        }
        if (!this._messageHandler)
            throw new Error("bridgedCamera message handler not found");
        zcout("Using bridged camera source");
    }
    static IsSupported() {
        var _a, _b;
        let hasWebkitBridge = !!((_b = (_a = window.webkit) === null || _a === void 0 ? void 0 : _a.messageHandlers) === null || _b === void 0 ? void 0 : _b.bridgedCamera);
        let hasAndroidBridge = AndroidBridgeMessageHandler.IsSupported();
        return hasWebkitBridge || hasAndroidBridge;
    }
    destroy() {
        this.pause();
        deleteCameraSource(this._impl);
    }
    pause() {
        this._isPaused = true;
        let p = Pipeline.get(this._pipeline);
        if (p && p.currentCameraSource === this)
            p.currentCameraSource = undefined;
        this._decodeBuffers = [];
        this._stopDeviceMotion();
        if (this._isAndroid)
            BridgedCameraSource._resetCount++;
        this._messageHandler.postMessage("stop");
    }
    start() {
        var _a;
        let p = Pipeline.get(this._pipeline);
        if (p && p.currentCameraSource !== this) {
            (_a = p.currentCameraSource) === null || _a === void 0 ? void 0 : _a.pause();
            p.currentCameraSource = this;
        }
        this._textureSizes.clear();
        this._isPaused = false;
        this._startDeviceMotion();
        let options = {};
        if (p) {
            options = Object.assign(Object.assign({}, BridgedWorldTracker.SharedInstance().getCombinedOptionsForPipeline(p)), BridgedD3Tracker.SharedInstance().getCombinedOptionsForPipeline(p));
        }
        if (!this._isAndroid)
            BridgedCameraSource._resetCount++;
        this._messageHandler.postMessage(`start${this._convertToBitmask(options)}:${this._profile}`);
    }
    getFrame(currentlyProcessing) {
        var _a, _b;
        let rotation = cameraRotationForScreenOrientation(false);
        if (rotation != this._cameraToScreenRotation) {
            (_b = (_a = Pipeline.get(this._pipeline)) === null || _a === void 0 ? void 0 : _a.sendCameraToScreenRotationToWorker) === null || _b === void 0 ? void 0 : _b.call(_a, rotation);
            this._cameraToScreenRotation = rotation;
        }
        if (currentlyProcessing) {
            // zcout("BridgedCameraSource: Worker already processing a frame, not requesting another");
            return;
        }
        this._processFrame();
    }
    _processFrame() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this._frameInFlight) {
                // zcout("BridgedCameraSource: Frame request already in flight, not requesting another");
                return;
            }
            traceStart("BCS: processFrame - request frameData");
            this._frameInFlight = true;
            let msgReply;
            let dataPromise = this._messageHandler.postMessage("frameData");
            traceEnd();
            try {
                msgReply = yield dataPromise;
            }
            catch (ex) {
                this._frameInFlight = false;
                return;
            }
            this._frameInFlight = false;
            if (!msgReply) {
                // zcout("BridgedCameraSource: No new frame data to process");
                return;
            }
            let decodeBuffer;
            let replyArrayBuffer;
            if (typeof msgReply === "object" && (msgReply instanceof ArrayBuffer)) {
                replyArrayBuffer = msgReply;
            }
            else if (typeof msgReply === 'string' && hasBridgeArrayBufferHeader(msgReply)) {
                traceStart("BCS: processFrame - convert string to ArrayBuffer");
                decodeBuffer = this._decodeBuffers.pop();
                while (decodeBuffer) {
                    if ((msgReply.length * 2) <= decodeBuffer.byteLength)
                        break;
                    decodeBuffer = this._decodeBuffers.pop();
                }
                if (!decodeBuffer) {
                    // Add some padding so moderate increases in message size (new planes, 
                    // boundary points, etc) will be able to reuse the same buffer
                    decodeBuffer = new ArrayBuffer(msgReply.length * 2 + 1024);
                }
                replyArrayBuffer = decodeBuffer;
                ArrayBufferFromString(replyArrayBuffer, msgReply);
                traceEnd();
            }
            else {
                zcout("BridgedCameraSource: Unexpected reply to frameData message");
                return;
            }
            if (this._isPaused) {
                zcout("BridgedCameraSource: Dropping frame - source paused");
                return;
            }
            let pipeline = Pipeline.get(this._pipeline);
            if (!pipeline)
                return;
            traceStart("BCS: processFrame - frameData received");
            traceStart("parseBridgeArrayBuffer");
            let frameData = parseBridgeArrayBuffer(replyArrayBuffer);
            traceEnd();
            if (!frameData) {
                zcout("BridgedCameraSource: Failed to parse frameData reply ArrayBuffer");
                traceEnd(); // BCS: processFrame - frameData received
                return;
            }
            const frameInfo = frameData.json;
            if (frameInfo.cameraModel) {
                this._cameraModel.set(frameInfo.cameraModel);
            }
            if (!('session_number' in frameInfo.wtData)) {
                // Fill in session_number with the fallback value
                frameInfo.wtData.session_number = BridgedCameraSource._resetCount;
            }
            traceStart(`Prepare CameraFrameInfo ${frameInfo.frameNo || 0}`);
            let numDataBytes = frameInfo.dataWidth * frameInfo.dataHeight;
            let pixels = pipeline.cameraPixelArrays.pop();
            while (pixels) {
                if (pixels.byteLength === numDataBytes)
                    break;
                pixels = pipeline.cameraPixelArrays.pop();
            }
            if (!pixels) {
                pixels = new ArrayBuffer(numDataBytes);
            }
            let pixelBytes = new Uint8Array(pixels);
            pixelBytes.set(frameData.uint8ArrayForBinaryBufferOffset(frameInfo.dataBinaryBufferOffset, frameInfo.dataBinaryBufferLength));
            let yData;
            let uvData;
            let vData;
            let uvBinaryBufferOffset = frameInfo.previewBinaryBufferOffset;
            if (frameInfo.dataWidth === frameInfo.yWidth && frameInfo.dataHeight === frameInfo.yHeight) {
                // Y isn't part of the preview bytes, it's shared with the data
                yData = frameData.uint8ArrayForBinaryBufferOffset(frameInfo.dataBinaryBufferOffset, frameInfo.dataBinaryBufferLength);
            }
            else {
                // Common case - previewData has the Y plane first
                const yBytes = frameInfo.yWidth * frameInfo.yHeight;
                yData = frameData.uint8ArrayForBinaryBufferOffset(frameInfo.previewBinaryBufferOffset, yBytes);
                uvBinaryBufferOffset += yBytes;
            }
            if (frameInfo.uvLayout === UvLayout.PLANAR_UV) {
                const uOffset = uvBinaryBufferOffset;
                const uBytes = frameInfo.uvWidth * frameInfo.uvHeight;
                const vOffset = uvBinaryBufferOffset + uBytes;
                uvData = frameData.uint8ArrayForBinaryBufferOffset(uOffset, uBytes);
                vData = frameData.uint8ArrayForBinaryBufferOffset(vOffset, uBytes);
            }
            else {
                // Interleaved
                const uvBytes = frameInfo.uvWidth * frameInfo.uvHeight * 2;
                uvData = frameData.uint8ArrayForBinaryBufferOffset(uvBinaryBufferOffset, uvBytes);
            }
            let info = {
                data: pixels,
                texture: undefined,
                dataWidth: frameInfo.dataWidth,
                dataHeight: frameInfo.dataHeight,
                cameraModel: this._cameraModel,
                cameraToDevice: this._cameraToDeviceTransform,
                cameraSource: this,
                cameraSourceData: {
                    decodeBuffer,
                    frameData,
                    frameInfo,
                    yData,
                    uvData,
                    vData
                },
                userFacing: false
            };
            traceEnd(); // Prepare CameraFrameInfo
            traceStart("Send to worker");
            let token = pipeline.registerToken(info);
            pipeline.sendDataToWorker(info.data, token, info.dataWidth, info.dataHeight, info.userFacing, info.cameraToDevice, info.cameraModel, performance.now() / 1000.0);
            info.data = undefined;
            traceEnd();
            traceEnd(); // BCS: processFrame - frameData received
        });
    }
    uploadGL(info) {
        if (!info || !(info.data) || !(info.cameraSourceData))
            return;
        const cameraSourceData = info.cameraSourceData;
        if (info.texture)
            return;
        const pipeline = Pipeline.get(this._pipeline);
        if (!pipeline)
            return;
        const gl = pipeline === null || pipeline === void 0 ? void 0 : pipeline.glContext;
        if (!gl)
            return;
        let texture = pipeline.getVideoTexture();
        if (!texture)
            return;
        let { yWidth, yHeight, uvWidth, uvHeight, uvLayout } = cameraSourceData.frameInfo;
        if (!yWidth || !yHeight || !uvWidth || !uvHeight)
            return;
        if (uvLayout === undefined) {
            uvLayout = 0;
        }
        let yPlane = {
            width: yWidth,
            height: yHeight,
            data: cameraSourceData.yData
        };
        let uvPlane = {
            width: uvWidth,
            height: uvHeight,
            data: cameraSourceData.uvData,
            dataVPlane: cameraSourceData.vData
        };
        let knownSize = this._textureSizes.get(texture);
        if (!knownSize || knownSize[0] != yPlane.width || knownSize[1] != yPlane.height) {
            // Resize the texture to match
            traceStart("Resize camera texture");
            const prevTex = gl.getParameter(gl.TEXTURE_BINDING_2D);
            gl.bindTexture(gl.TEXTURE_2D, texture);
            gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, yPlane.width, yPlane.height, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
            gl.bindTexture(gl.TEXTURE_2D, prevTex);
            this._textureSizes.set(texture, [yPlane.width, yPlane.height]);
            traceEnd();
            zcout(`BridgedCameraSource: Resized a texture, elements in map ${this._textureSizes.size}`);
        }
        // Run the YUV -> RGBA conversion
        traceStart("YUV to Texture");
        if (!this._yuvConversion)
            this._yuvConversion = new YUVConversionGL(gl);
        this._yuvConversion.yuvToTexture(yPlane, uvPlane, uvLayout, texture);
        traceEnd();
        // Set the texture in the CameraFrameInfo
        info.texture = texture;
    }
    recycle(info) {
        var _a;
        // Only bother keeping decode buffers for reuse if we're not paused
        if (this._isPaused)
            return;
        // Don't want to accumulate too many
        if (this._decodeBuffers.length >= 2)
            return;
        let recycleBuffer = (_a = info.cameraSourceData) === null || _a === void 0 ? void 0 : _a.decodeBuffer;
        if (recycleBuffer)
            this._decodeBuffers.push(recycleBuffer);
    }
    setProfile(p) {
        if (this._profile === p)
            return;
        this._profile = p;
        if (!this._isPaused) {
            this._messageHandler.postMessage(`setProfile${this._profile}`);
        }
    }
    setCustomAnchorPose(anchor, pose_version, pose) {
        let message = `setCustomAnchorPose${anchor}:${pose_version}:`;
        this._anchorPoseFloat32.set(pose);
        message += String.fromCharCode(...this._anchorPoseUint16);
        this._messageHandler.postMessage(message);
    }
    resetTracking() {
        BridgedCameraSource._resetCount++;
        this._messageHandler.postMessage('resetTracking');
    }
    _convertToBitmask(options) {
        let bitmask = 0;
        if (options.worldTracking)
            bitmask |= 1;
        if (options.horizontalPlaneDetection)
            bitmask |= 2;
        if (options.verticalPlaneDetection)
            bitmask |= 4;
        if (options.tracksData)
            bitmask |= 8;
        if (options.projectionsData)
            bitmask |= 16;
        if (options.meshAnchors)
            bitmask |= 32;
        if (options.d3)
            bitmask |= 64;
        if (options.d3MaxResolution)
            bitmask |= 128;
        return bitmask;
    }
    optionsUpdated(options) {
        this._messageHandler.postMessage(`setOptions${this._convertToBitmask(options)}`);
    }
    _startDeviceMotion() {
        window.addEventListener("devicemotion", this._deviceMotionListener);
        window.addEventListener("deviceorientation", this._deviceOrientationListener);
    }
    _stopDeviceMotion() {
        window.removeEventListener("devicemotion", this._deviceMotionListener);
        window.removeEventListener("deviceorientation", this._deviceOrientationListener);
    }
}
BridgedCameraSource.DEFAULT_DEVICE_ID = "Simulated Default Device ID: a908df7f-5661-4d20-b227-a1c15d2fdb4b";
// Fallback for session_number if native side doesn't fill it in
BridgedCameraSource._resetCount = 0;
