import { Component, ComponentConstructor, Object3D } from '@wonderlandengine/api';
/**
 * Convert a TypeScript syntactic sugar enum to a list of keys.
 *
 * This method filters the reverse mapping from value to key.
 *
 * @note Index mapping must be contiguous and 0-based.
 *
 * @param e Enum to retrieve the keys from
 * @returns The list of keys
 */
export declare function enumStringKeys<T extends object>(e: T): (keyof T)[];
/**
 * Calculates the world space radius of an Object3D hierarchy.
 *
 * The function returns the radius of a bounding sphere that encloses all mesh components
 * in the hierarchy of the provided Object3D. This can be useful for visibility testing,
 * collision detection, or spatial queries within a 3D environment.
 *
 * @param object The root Object3D of the hierarchy to calculate the world space radius for.
 * @returns The radius of the world space bounding sphere.
 *
 * @example
 * // Assuming 'rootObject' is the root of your Object3D hierarchy with mesh components
 * const worldRadius = radiusHierarchy(rootObject);
 * console.log(`World space radius: ${worldRadius}`);
 */
export declare const radiusHierarchy: (object: Object3D) => number;
/**
 * Recursively sets the active state of the given object and all its children.
 * @param object The object to set the active state of.
 * @param active The state to set.
 *
 * @example
 * ```js
 * // turn off all components on this object and its children
 * setComponentsActive(this.someObject, undefined, false);
 * ```
 */
export declare function setComponentsActive(object: Object3D, active: boolean, typeClass?: ComponentConstructor): void;
export declare function componentError(component: Component, msg: string): string;
