/// <reference types="webxr" />
import { Component, Emitter, Object3D } from '@wonderlandengine/api';
/**
 * Typename for a `PlayerControllerInput` component.
 *
 * Use this to create your own input component that can
 * be retrieved by {@link PlayerController}:
 *
 * ```ts
 * import {PlayerControllerInput, PlayerControllerInputTypename} fromn '@wonderlandengine/interaction';
 *
 * export class CustomPlayerInput implements PlayerControllerInput {
 *     static TypeName = PlayerControllerInputTypename;
 *     ...
 * }
 * ```
 */
export declare const InteractorInputTypename = "interactor-input";
/**
 * Defines an interface for bridging inputs with {@link Interactor}.
 *
 * For custom inputs, implement this interface, e.g.,
 *
 * @example
 * ```js
 * export class CustomInput extends Component implements InteractorInput {
 *     static TypeName = InteractorInputTypename;
 *
 *     onGrabStart: Emitter;
 *     onGrabEnd: Emitter;
 *
 *     update() {
 *         // Notify `onGrabStart` and `onGrabEnd`.
 *         // Could be an event from the network, anything.
 *     }
 *
 *     \/** @override *\/
 *     get xrPose(): XRPose {
 *         return this._xrPose;
 *     }
 * }
 * ```
 *
 * The custom bridge doesn't need to be a Wonderland Engine component.
 */
export interface InteractorInput extends Component {
    /** Notify once the grab must start. */
    onGrabStart: Emitter;
    /** Notify once the grab must end. */
    onGrabEnd: Emitter;
    /** Current controller XR pose. */
    get xrPose(): XRPose | null;
}
/** Semantic mapping for XR controller button. */
export declare enum XRButton {
    Trigger = 0,
    Grip = 1,
    Joystick = 2,
    PrimaryButton = 3,
    SecondaryButton = 4
}
/** List of string keys for {@link XRButton}. */
export declare const XRButtonNames: ("Trigger" | "Grip" | "Joystick" | "PrimaryButton" | "SecondaryButton")[];
/**
 * Default inputs for {@link Interactor}.
 *
 * Currently supports:
 * - VR controllers
 *
 * For fully custom inputs, derive {@link InteractorInput} instead.
 */
export declare class DefaultInteractorInput extends Component implements InteractorInput {
    static TypeName: string;
    /** Object with a native input component.*/
    inputObject: Object3D;
    grab: XRButton;
    onGrabStart: Emitter<void[]>;
    onGrabEnd: Emitter<void[]>;
    private _input;
    private _grabPressed;
    private _xrPose;
    /** @hidden */
    private readonly _onPreRender;
    getInputSourceXR(): XRInputSource | null;
    /** @override */
    onActivate(): void;
    onDeactivate(): void;
    /** @override */
    update(): void;
    get xrPose(): XRPose | null;
}
