import { Component } from '@wonderlandengine/api';
import { Interactor } from './interactor.js';
/**
 * Visual state to apply to an interactor.
 */
export declare enum InteractorVisualState {
    /** Unspecified. */
    None = 0,
    /** Interactor mesh should be set to visible. */
    Visible = 1,
    /** Interactor mesh should be hidden. */
    Hidden = 2
}
/** List of string keys for {@link InteractorVisualState}. */
export declare const InteractorVisualStateNames: ("None" | "Visible" | "Hidden")[];
/**
 * Describe how the {@link Interactor} interacts with a {@link GrabPoint}.
 */
export declare enum GrabSearchMode {
    /**
     * Interaction occurs if the distance with the grab point
     * is in range of {@link GrabPoint.maxDistance}.
     */
    Distance = 0,
    /**
     * Interaction occurs if the grab point collision or physx collider
     * overlaps with the interactor.
     */
    Overlap = 1
}
/** List of string keys for {@link GrabSearchMode}. */
export declare const GrabSearchModeNames: ("Distance" | "Overlap")[];
/**
 * Describe how the {@link Grabbable} behave once the interaction starts
 * with the {@link GrabPoint}.
 */
export declare enum GrabSnapMode {
    /**
     * Grabbable is grabbed at the point of interaction,
     * doesn't snap to the grab point.
     */
    None = 0,
    /**
     * Grabbable is snapped to the interactor, using the grab point
     * as position / rotation for anchoring.
     */
    PositionRotation = 1
}
/** List of string keys for {@link GrabSnapMode}. */
export declare const GrabSnapModeNames: ("None" | "PositionRotation")[];
/**
 * Link used to specify how / where a {@link Grabbable} is grabbed.
 */
export declare class GrabPoint extends Component {
    static TypeName: string;
    /** Snap mode, defaults to {@link GrabSnapMode.PositionRotation} */
    snap: GrabSnapMode;
    /**
     * Lerp value used during interaction.
     *
     * Smaller values make the grabbable more slowly snap to the interactor,
     * higher values make the snapping effect more instantaneous.
     *
     * @note This value is clamped in the range `[0, 1]`.
     *
     * Defaults to `0.75`.
     */
    snapLerp: number;
    /** Search mode, defaults to {@link GrabSearchMode.Distance} */
    searchMode: GrabSearchMode;
    /**
     * Maximum distance at which interaction can occur.
     *
     * @note This is only used if {@link searchMode} is set to {@link GrabSearchMode.Distance}.
     */
    maxDistance: number;
    /** If `true`, handle can be transfered to another interactor. */
    transferable: boolean;
    /** Visual state to apply to the interactor once interaction occurs. */
    interactorVisualState: InteractorVisualState;
    /** @hidden */
    _interactor: Interactor | null;
    /** Current interactor managing this instance. */
    get interactor(): Interactor | null;
}
