/// <reference path="../../src/types/global.d.ts" />
/// <reference types="webxr" />
import { Component, Emitter } from '@wonderlandengine/api';
import { mat4 } from 'gl-matrix';
import { ARProvider, ARSession, ITrackingMode, ImageScanningEvent, TrackingType } from '@wonderlandengine/ar-tracking';
import { loadZappar } from './zappar-module.js';
import type { FaceMesh, FaceTracker, ImageTracker, Pipeline } from '@zappar/zappar';
type ZapparImageTargetType = 'flat' | 'cylindrical' | 'conical';
/** A single detected horizontal plane produced by {@link ZapparWorldTracker}. */
export interface ZapparPlaneAnchor {
    /** Pose matrix of this plane in world space. */
    pose(cameraPose: Float32Array): Float32Array;
    /** Ordered 2-D boundary vertices on the plane surface. */
    readonly polygon: ReadonlyArray<readonly [number, number]>;
    /** Increments each time `polygon` is updated. */
    readonly polygonVersion: number;
}
/**
 * Minimal interface for the Zappar `WorldTracker` class available in
 * `@zappar/zappar >= 4.x`.  Only the API surface used by this package is
 * declared here.
 */
export interface ZapparWorldTracker {
    enabled: boolean;
    horizontalPlaneDetectionEnabled: boolean;
    verticalPlaneDetectionEnabled: boolean;
    /** Map of all currently detected plane anchors keyed by their ID. */
    readonly planes: ReadonlyMap<string, ZapparPlaneAnchor>;
    readonly worldAnchor: {
        pose(cameraPose: Float32Array): Float32Array;
    };
    readonly groundAnchor: {
        pose(cameraPose: Float32Array): Float32Array;
    };
}
export interface ZapparImageTargetOptions {
    name: string;
    type?: ZapparImageTargetType;
    physicalWidthInMeters?: number;
    metadata?: unknown;
}
interface ZapparImageTargetDescriptor {
    name: string;
    type: ZapparImageTargetType;
    geometry: ImageScanningEvent['imageTargets'][0]['geometry'];
    properties: ImageScanningEvent['imageTargets'][0]['properties'];
    metadata: unknown;
}
/**
 * ARProvider implementation backed by the Zappar Universal AR JavaScript SDK.
 */
export declare class ZapparProvider extends ARProvider {
    private static _cvWorkerConfigured;
    private static _cvWorker;
    /** Mirror Zappar THREE.js `CameraPoseMode` for SLAM pose output. */
    slamPoseMode: 'default' | 'attitude' | 'anchor-origin';
    private _xrSession;
    private _gl;
    private _zappar;
    private pipeline;
    private cameraSource;
    private instantTracker;
    private _faceTracker;
    private _faceMesh;
    private _faceResourcesPromise;
    private _imageTracker;
    private _preferredCameraUserFacing;
    private _cameraSourceUserFacing;
    private _imageTargetDescriptors;
    private readonly _imageTargetsChanged;
    private _worldTracker;
    /** Set to `true` when a component requests world-tracker before the session starts. */
    private _worldTrackerRequested;
    private cameraStarted;
    private hasInitializedAnchor;
    private _anchorWarmupFramesRemaining;
    /**
     * When `true`, {@link updateTracking} keeps calling
     * `setAnchorPoseFromCameraOffset` every frame so the world anchor
     * continuously follows the camera centre. Set via
     * {@link startUserPlacement}; cleared by {@link placeInstantAnchor}.
     */
    private _userPlacementMode;
    private preRenderRegistered;
    private _slamStateValid;
    private readonly _slamProjectionMatrix;
    private readonly _slamAnchorMatrix;
    private _slamFrameNumber;
    private _videoTextureUnit;
    private _videoTextureProgram;
    private _videoTextureUniform;
    private _videoTextureTransformUniform;
    private _videoTextureBindErrorLogged;
    private _missingCameraTextureFrames;
    private _missingCameraTextureWarningLogged;
    private readonly _slamCameraMatrix;
    private readonly _slamCameraPosition;
    private readonly _slamCameraRotation;
    private readonly _debugCameraPosition;
    private readonly _debugAnchorPosition;
    private readonly _debugCameraPositionDelta;
    private readonly _debugAnchorPositionDelta;
    private _debugLastSampleFrameNumber;
    private readonly _debugLastSampleCameraPosition;
    private readonly _debugLastSampleAnchorPosition;
    private _zapparDebugLogIntervalId;
    private _preRenderErrorLogged;
    private get isVerboseDebugLoggingEnabled();
    static Name: string;
    get name(): string;
    get supportsInstantTracking(): boolean;
    get onImageTargetsChanged(): Emitter<void[]>;
    get xrSession(): XRSession | null;
    /**
     * Hint camera facing preference for the next/active session.
     * - `true`: user/front camera
     * - `false`: rear/back camera
     * - `null`: Zappar default
     */
    setPreferredCameraUserFacing(userFacing: boolean | null): void;
    static registerTrackingProviderWithARSession(arSession: ARSession): ZapparProvider;
    private constructor();
    startSession(): Promise<void>;
    private ensurePreRenderRegistered;
    private startZapparDebugLogging;
    private stopZapparDebugLogging;
    private ensureZapparLoaded;
    /** Used by tracking modes that need the Zappar namespace. */
    ensureZapparNamespace(): Promise<Awaited<ReturnType<typeof loadZappar>>>;
    private configureCvWorkerIfNeeded;
    private ensurePipeline;
    private ensureCameraSourcePreference;
    getPipeline(): Pipeline;
    ensureFaceResources(): Promise<void>;
    getFaceTracker(): FaceTracker;
    getFaceMesh(): FaceMesh;
    ensureImageTracker(): ImageTracker;
    registerImageTarget(source: string | ArrayBuffer, options: ZapparImageTargetOptions): Promise<void>;
    private _inferImageTargetType;
    private _buildImageTargetGeometry;
    getImageScanningEvent(): ImageScanningEvent;
    getImageTargetDescriptors(): ReadonlyArray<ZapparImageTargetDescriptor>;
    getImageTargetDescriptor(index: number): ZapparImageTargetDescriptor | undefined;
    /**
     * Access the active {@link ZapparWorldTracker} instance, or `null` when
     * plane tracking has not been requested or is unavailable.
     *
     * The instance is only non-null once {@link enableWorldTracker} has been
     * called **and** an AR session is running.
     */
    get worldTracker(): ZapparWorldTracker | null;
    /**
     * Opt in to Zappar plane tracking via the `WorldTracker` API introduced in
     * `@zappar/zappar >= 4.x`.
     *
     * Call this once (e.g. from a component's `start()` hook) before or after
     * a session starts.  If the installed SDK does not expose `WorldTracker`
     * (i.e. < 4.x) a warning is logged and the method returns `null`.
     *
     * The tracker is disabled again when the session ends and re-enabled on
     * the next `startSession()` call.
     *
     * @returns The {@link ZapparWorldTracker} instance, or `null` on failure.
     */
    enableWorldTracker(): ZapparWorldTracker | null;
    private _createWorldTracker;
    private ensureCameraRunning;
    private onVisibilityChange;
    private onPreRender;
    /** Latest projection matrix for the SLAM camera (valid only if {@link hasSlamTrackingState} is true). */
    get slamProjectionMatrix(): Readonly<Float32Array> | null;
    /** Latest camera pose matrix for the SLAM camera (valid only if {@link hasSlamTrackingState} is true). */
    get slamCameraPoseMatrix(): Readonly<mat4> | null;
    /** Latest anchor pose matrix (valid only if {@link hasSlamTrackingState} is true). */
    get slamAnchorPoseMatrix(): Readonly<mat4> | null;
    get hasSlamTrackingState(): boolean;
    /**
     * `true` while the instant world anchor is being continuously repositioned
     * in front of the camera (either during warmup or during user-placement
     * mode). `false` once the anchor has been locked.
     */
    get isPlacingInstantAnchor(): boolean;
    /**
     * Enter user-placement mode: the instant world anchor tracks 5 m in front
     * of the camera every frame until {@link placeInstantAnchor} is called.
     *
     * Typically called by a placement-UI component immediately after session
     * start so the user can choose where to lock the world origin.
     */
    startUserPlacement(): void;
    /**
     * Lock the instant world anchor at its current position and exit placement
     * mode. After this call {@link isPlacingInstantAnchor} returns `false` and
     * the anchor no longer follows the camera.
     */
    placeInstantAnchor(): void;
    get slamFrameNumber(): number;
    private bindVideoTextureForSkyMaterial;
    updateTracking(): void;
    endSession(): Promise<void>;
    load(): Promise<void>;
    /** Whether this provider supports given tracking type */
    supports(type: TrackingType): boolean;
    /** Create a tracking implementation */
    createTracking(type: TrackingType, component: Component): ITrackingMode;
}
export {};
