/// <reference types="webxr" />
import { Component, Object3D, Emitter, WonderlandEngine } from '@wonderlandengine/api';
import { vec3 } from 'gl-matrix';
import { CursorTarget } from './cursor-target.js';
import { HitTestLocation } from './hit-test-location.js';
export type EventTypes = PointerEvent | MouseEvent | XRInputSourceEvent;
export interface CursorEvents<T, K> {
    onHover: Emitter<[T, K, EventTypes?]>;
    onUnhover: Emitter<[T, K, EventTypes?]>;
    onClick: Emitter<[T, K, EventTypes?]>;
    onMove: Emitter<[T, K, EventTypes?]>;
    onDown: Emitter<[T, K, EventTypes?]>;
    onUp: Emitter<[T, K, EventTypes?]>;
}
/** Global target for {@link Cursor} */
declare class CursorTargetEmitters<T, K> implements CursorEvents<T, K> {
    /** Emitter for events when the target is hovered */
    onHover: Emitter<[T, K, (EventTypes | undefined)?]>;
    /** Emitter for events when the target is unhovered */
    onUnhover: Emitter<[T, K, (EventTypes | undefined)?]>;
    /** Emitter for events when the target is clicked */
    onClick: Emitter<[T, K, (EventTypes | undefined)?]>;
    /** Emitter for events when the cursor moves on the target */
    onMove: Emitter<[T, K, (EventTypes | undefined)?]>;
    /** Emitter for events when the user pressed the select button on the target */
    onDown: Emitter<[T, K, (EventTypes | undefined)?]>;
    /** Emitter for events when the user unpressed the select button on the target */
    onUp: Emitter<[T, K, (EventTypes | undefined)?]>;
}
/**
 * 3D cursor for desktop/mobile/VR.
 *
 * Implements a ray-casting cursor into the scene. To react to
 * clicking/hover/unhover/cursor down/cursor up/move use a
 * [cursor-target](#cursor-target).
 *
 * For VR, the ray is cast in direction of
 * [this.object.getForward()](/jsapi/object/#getforward). For desktop and mobile, the
 * forward vector is inverse-projected to account for where on screen the user clicked.
 *
 * `.globalTarget` can be used to call callbacks for all objects, even those that
 * do not have a cursor target attached, but match the collision group.
 *
 * `.hitTestTarget` can be used to call callbacks WebXR hit test results,
 *
 * See [Animation Example](/showcase/animation).
 */
export declare class Cursor extends Component {
    static TypeName: string;
    static Dependencies: (typeof HitTestLocation)[];
    static onRegister(engine: WonderlandEngine): void;
    private _collisionMask;
    private _onDeactivateCallbacks;
    private _input;
    private _origin;
    private _cursorObjScale;
    private _direction;
    private _projectionMatrix;
    private _viewComponent;
    private _isDown;
    private _lastIsDown;
    private _arTouchDown;
    private _lastPointerPos;
    private _lastCursorPosOnTarget;
    private _hitTestLocation;
    private _hitTestObject;
    private _onSessionStartCallback;
    /**
     * Whether the cursor (and cursorObject) is visible, i.e. pointing at an object
     * that matches the collision group
     */
    visible: boolean;
    /** Currently hovered object */
    hoveringObject: Object3D | null;
    /** CursorTarget component of the currently hovered object */
    hoveringObjectTarget: CursorTarget | null;
    /** Whether the cursor is hovering reality via hit-test */
    hoveringReality: boolean;
    /**
     * Global target lets you receive global cursor events on any object.
     */
    globalTarget: CursorTargetEmitters<Object3D, Cursor>;
    /**
     * Hit test target lets you receive cursor events for "reality", if
     * `useWebXRHitTest` is set to `true`.
     *
     * @example
     * ```js
     * cursor.hitTestTarget.onClick.add((hit, cursor) => {
     *     // User clicked on reality
     * });
     * ```
     */
    hitTestTarget: CursorTargetEmitters<XRHitTestResult | null, Cursor>;
    /** World position of the cursor */
    cursorPos: Float32Array;
    /** Collision group for the ray cast. Only objects in this group will be affected by this cursor. */
    collisionGroup: number;
    /** (optional) Object that visualizes the cursor's ray. */
    cursorRayObject: Object3D | null;
    /** Axis along which to scale the `cursorRayObject`. */
    cursorRayScalingAxis: number;
    /** (optional) Object that visualizes the cursor's hit location. */
    cursorObject: Object3D | null;
    /** Handedness for VR cursors to accept trigger events only from respective controller. */
    handedness: number | string;
    /** Object that has an input component. */
    inputObject: Object3D | null;
    /** Object that has a view component. */
    viewObject: Object3D | null;
    /** Mode for raycasting, whether to use PhysX or simple collision components */
    rayCastMode: number | string;
    /** Maximum distance for the cursor's ray cast. */
    maxDistance: number;
    /** Whether to set the CSS style of the mouse cursor on desktop */
    styleCursor: boolean;
    /**
     * Use WebXR hit-test if available.
     *
     * Attaches a hit-test-location component to the cursorObject, which will be used
     * by the cursor to send events to the hitTestTarget with HitTestResult.
     */
    useWebXRHitTest: boolean;
    _onViewportResize: () => void;
    start(): void;
    onActivate(): void;
    _setCursorRayTransform(hitPosition: vec3): void;
    _setCursorVisibility(visible: boolean): void;
    update(): void;
    private notify;
    private hoverBehaviour;
    /**
     * Setup event listeners on session object
     * @param s WebXR session
     *
     * Sets up 'select' and 'end' events.
     */
    setupVREvents(s: XRSession): void;
    onDeactivate(): void;
    onDestroy(): void;
    /** 'select' event listener */
    onSelect(e: XRInputSourceEvent): void;
    /** 'selectstart' event listener */
    onSelectStart(e: XRInputSourceEvent): void;
    /** 'selectend' event listener */
    onSelectEnd(e: XRInputSourceEvent): void;
    /** 'pointermove' event listener */
    onPointerMove(e: PointerEvent): void;
    /** 'click' event listener */
    onClick(e: MouseEvent): void;
    /** 'pointerdown' event listener */
    onPointerDown(e: PointerEvent): void;
    /** 'pointerup' event listener */
    onPointerUp(e: PointerEvent): void;
    /**
     * Update mouse position in non-VR mode and raycast for new position
     * @returns @ref RayHit for new position.
     */
    private updateMousePos;
    private updateDirection;
    private applyTransformAndProjectDirection;
    private applyTransformToDirection;
    private rayCast;
}
export {};
