import { Component, Emitter, WonderlandEngine } from '@wonderlandengine/api';
import { AudioChannel, AudioManager, PlayState } from './audio-manager.js';
export declare enum PanningType {
    None = 0,
    Regular = 1,
    Hrtf = 2
}
/**
 * Loads the given audio into a AudioBuffer.
 *
 * @param source Path to the file that should be decoded
 * @returns A Promise that fulfills once the audio is decoded
 */
export declare function loadAudio(source: string): Promise<AudioBuffer>;
/**
 * Represents an audio src in the Wonderland Engine, allowing playback of audio files.
 */
export declare class AudioSource extends Component {
    /**
     * The type name for this component.
     */
    static TypeName: string;
    static onRegister(engine: WonderlandEngine): void;
    /** Path to the audio file that should be played. */
    src: string;
    /**
     * Volume of the audio source.
     *
     * @remarks This will only take effect audio that has not started playing yet. Is the audio already playing, use
     * setVolumeDuringPlayback()
     * @see setVolumeDuringPlayback
     */
    volume: number;
    /** Whether to loop the sound. */
    loop: boolean;
    /** Whether to autoplay the sound. */
    autoplay: boolean;
    /**
     * Select the panning method.
     *
     * @warning Enabling HRTF (Head-Related Transfer Function) is computationally more intensive than regular panning!
     */
    spatial: PanningType;
    /**
     * Set this property if the object will never move.
     * Disabling position updates each frame saves CPU time.
     */
    isStationary: boolean;
    /** The distance model used for spatial audio. */
    distanceModel: DistanceModelType;
    /** The maximum distance for audio falloff. */
    maxDistance: number;
    /** The reference distance for audio falloff. */
    refDistance: number;
    /** The rolloff factor for audio falloff. */
    rolloffFactor: number;
    /** The inner angle of the audio cone. */
    coneInnerAngle: number;
    /** The outer angle of the audio cone. */
    coneOuterAngle: number;
    /** The outer gain of the audio cone. */
    coneOuterGain: number;
    /**
     * The emitter will notify all subscribers when a state change occurs.
     * @see PlayState
     */
    readonly emitter: Emitter<[PlayState]>;
    private _pannerOptions;
    private _buffer;
    private _pannerNode;
    private _audioNode;
    private _isPlaying;
    private _time;
    private readonly _gainNode;
    /**
     * Initializes the audio src component.
     * If `autoplay` is enabled, the audio will start playing as soon as the file is loaded.
     */
    start(): Promise<void>;
    setAudioChannel(am: AudioManager, channel: AudioChannel): void;
    /**
     * Plays the audio associated with this audio src.
     *
     * @param buffer Optional parameter that will set the raw audio buffer that should be played. Defaults to internal audio buffer that is set with given audio path.
     * @remarks Is this audio-source currently playing, playback will be restarted.
     */
    play(buffer?: AudioBuffer): Promise<void>;
    /**
     * Stops the audio associated with this audio src.
     */
    stop(): void;
    /**
     * Checks if the audio src is currently playing.
     */
    get isPlaying(): boolean;
    /**
     * Changes the volume during playback.
     * @param v Volume that source should have.
     * @param t Optional parameter that specifies the time it takes for the volume to reach its specified value in
     * seconds (Default is 0).
     */
    setVolumeDuringPlayback(v: number, t?: number): void;
    /**
     * Change out the source.
     *
     * @param path Path to the audio file.
     */
    changeAudioSource(path: string): Promise<void>;
    /**
     * Called when the component is deactivated.
     * Stops the audio playback.
     */
    onDeactivate(): void;
    /**
     * Called when the component is destroyed.
     * Stops the audio playback and removes the src from cache.
     */
    onDestroy(): void;
    private _update;
    /**
     * @deprecated Use {@link #volume} instead
     */
    set maxVolume(v: number);
    /**
     * @deprecated Use {@link #volume} instead
     */
    get maxVolume(): number;
    private _updateSettings;
    private _distanceModelSelector;
}
