/// <reference types="webxr" />
import { Component, Object3D, Mesh, Skin, Material } from '@wonderlandengine/api';
/**
 * Easy hand tracking through the WebXR Device API
 * ["Hand Input" API](https://immersive-web.github.io/webxr-hand-input/).
 *
 * Allows displaying hands either as sphere-joints or skinned mesh.
 *
 * To react to grabbing, use `this.isGrabbing()`. For other gestures, refer
 * to `this.joints` - an array of [Object3D](/jsapi/object3d) and use the joint
 * indices listed [in the WebXR Hand Input specification](https://immersive-web.github.io/webxr-hand-input/#skeleton-joints-section).
 *
 * It is often desired to use either hand tracking or controllers, not both.
 * This component provides `deactivateChildrenWithoutPose` to hide the hand
 * tracking visualization if no pose is available and `controllerToDeactivate`
 * for disabling another object once a hand tracking pose *is* available.
 * Outside of XR sessions, tracking or controllers are neither enabled nor disabled
 * to play well with the [vr-mode-active-switch](#vr-mode-active-switch) component.
 *
 * **Requirements:**
 *  - To use hand-tracking, enable "joint tracking" in `chrome://flags` on
 *    Oculus Browser for Oculus Quest/Oculus Quest 2.
 *
 * See [Hand Tracking Example](/showcase/hand-tracking).
 */
export declare class HandTracking extends Component {
    static TypeName: string;
    /** Handedness determining whether to receive tracking input from right or left hand */
    handedness: string | number;
    /** (optional) Mesh to use to visualize joints */
    jointMesh: Mesh | null;
    /** Material to use for display. Applied to either the spawned skinned mesh or the joint spheres. */
    jointMaterial: Material | null;
    /** (optional) Skin to apply tracked joint poses to. If not present,
     * joint spheres will be used for display instead. */
    handSkin: Skin | null;
    /** Deactivate children if no pose was tracked */
    deactivateChildrenWithoutPose: boolean;
    /** Controller objects to activate including children if no pose is available */
    controllerToDeactivate: Object3D | null;
    init(): void;
    joints: Record<string, Object3D>;
    session: XRSession | null;
    hasPose: boolean;
    _childrenActive: boolean;
    start(): void;
    update(dt: number): void;
    setChildrenActive(active: boolean, object?: Object3D): void;
    isGrabbing(): boolean;
}
